library IEEE;
    use IEEE.STD_LOGIC_1164.ALL;
    use IEEE.NUMERIC_STD.ALL;

    use work.TIMER_PKG.ALL;

entity TIMER_SOC is
    port (
        sysclk_in   : std_logic;
        
        -- Key Controls
        keyUp_in    : in std_logic;
        keyDown_in  : in std_logic;
        keyLeft_in  : in std_logic;
        keyRight_in : in std_logic;
        keySet_in   : in std_logic;
        
        switches_in : in std_logic_vector(7 downto 0);
        
        -- 7 Segment
        digits_out   : out std_logic_vector(3 downto 0);
        segments_out : out std_logic_vector(7 downto 0);
        
        -- Leds
        leds_out     : out std_logic_vector(7 downto 0) := (others => '0')
    );
end TIMER_SOC;

architecture Behavioral of TIMER_SOC is
    constant SYSCLK_FREQ : positive := 100e6;
    constant SAMPLING_WIDTH : positive := 32;
    constant RESET_LENGTH : positive := SYSCLK_FREQ / 500 + 100;
    
    signal reset_cnt_i : integer range 0 to RESET_LENGTH := RESET_LENGTH;
    signal sampling_clock_i, clk10hz_i, clk100hz_i, clkTimer_i, reset_i : std_logic;
    signal deb_keyUp_i, deb_keyDown_i, deb_keyLeft_i, deb_keyRight_i, deb_keySet_i : std_logic;
    
    signal disp_dots_i, disp_brightness_i : std_logic_vector(3 downto 0);
    signal disp_data_i : std_logic_vector(15 downto 0);
    
    signal alarm_i : std_logic;

    signal alu_instruction_i : ALU_INSTRUCTION_TYPE;
    signal timer_is_zero_i : std_logic;
    signal timer_i : TIMER_VALUE_TYPE;
begin
-- controller
    my_controller: CONTROLLER port map (
        sysclk_in  => sysclk_in,
        clk10hz_in => clkTimer_i,
        reset_in   => reset_i,
        
        alu_instruction_out => alu_instruction_i,
        timer_is_zero_in    => timer_is_zero_i,
        
        keyUp_in    => deb_keyUp_i,
        keyDown_in  => deb_keyDown_i,
        keyLeft_in  => deb_keyLeft_i,
        keyRight_in => deb_keyRight_i,
        keySet_in   => deb_keySet_i,
        
        disp_highlight_out => disp_brightness_i,
        disp_dots_out => disp_dots_i,
        
        alarm_out => alarm_i
    );
    
    leds_out(6 downto 0) <= (6 => '0', others => alarm_i);

-- alu
    my_alu : ALU port map (
        sysclk_in => sysclk_in,
        reset_in  => reset_i,
        instruction_in => alu_instruction_i,
        timer_out => timer_i,
        timer_is_zero_out => timer_is_zero_i
    );

    disp_data_i <= STD_LOGIC_VECTOR(
        TO_UNSIGNED(timer_i.min, 4) & TO_UNSIGNED(timer_i.sec10, 4) &
        TO_UNSIGNED(timer_i.sec, 4) & TO_UNSIGNED(timer_i.tenth, 4)
    );

-- key debouncer
    my_deb_keyUp : DEBOUNCER generic map (
        SAMPLING_WIDTH => SAMPLING_WIDTH
    ) port map (
        sysclk_in => sysclk_in,
        sampling_clk_in => sampling_clock_i,
        reset_in  => reset_i,
        
        button_in => keyUp_in,
        pressed_out => deb_keyUp_i
    );

    my_deb_keyDown : DEBOUNCER generic map (
        SAMPLING_WIDTH => SAMPLING_WIDTH
    ) port map (
        sysclk_in => sysclk_in,
        sampling_clk_in => sampling_clock_i,
        reset_in  => reset_i,
        
        button_in => keyDown_in,
        pressed_out => deb_keyDown_i
    );

    my_deb_keyLeft : DEBOUNCER generic map (
        SAMPLING_WIDTH => SAMPLING_WIDTH
    ) port map (
        sysclk_in => sysclk_in,
        sampling_clk_in => sampling_clock_i,
        reset_in  => reset_i,
        
        button_in => keyLeft_in,
        pressed_out => deb_keyLeft_i
    );

    my_deb_keyRight : DEBOUNCER generic map (
        SAMPLING_WIDTH => SAMPLING_WIDTH
    ) port map (
        sysclk_in => sysclk_in,
        sampling_clk_in => sampling_clock_i,
        reset_in  => reset_i,
        
        button_in => keyRight_in,
        pressed_out => deb_keyRight_i
    );

    my_deb_keySet : DEBOUNCER generic map (
        SAMPLING_WIDTH => SAMPLING_WIDTH
    ) port map (
        sysclk_in => sysclk_in,
        sampling_clk_in => sampling_clock_i,
        reset_in  => reset_i,
        
        button_in => keySet_in,
        pressed_out => deb_keySet_i
    );

-- display driver
    my_display: DISPLAY generic map (
        BRIGHTNESS_BUST_FACTOR => 10,
        DIGITS_NO => 4
    ) port map (
        clk_in   => sampling_clock_i,
        reset_in => reset_i,

        data_in       => disp_data_i,
        dots_in       => disp_dots_i,
        brightness_in => disp_brightness_i,
        
        digits_out   => digits_out,
        segments_out => segments_out
    );

-- clock divider
    my_clkdiv_sampling: CLOCK_DIVIDER generic map (
        SOURCE_FREQUENCY => SYSCLK_FREQ,
        TARGET_FREQUENCY => 10000
    ) port map (
        clk_in   => sysclk_in,
        reset_in => reset_i,
        clk_out  => sampling_clock_i
    );

    my_clkdiv_100hz: CLOCK_DIVIDER generic map (
        SOURCE_FREQUENCY => SYSCLK_FREQ,
        TARGET_FREQUENCY => 100
    ) port map (
        clk_in   => sysclk_in,
        reset_in => reset_i,
        clk_out  => clk100hz_i
    );

    my_clkdiv_10hz: CLOCK_DIVIDER generic map (
        SOURCE_FREQUENCY => SYSCLK_FREQ,
        TARGET_FREQUENCY => 10
    ) port map (
        clk_in   => sysclk_in,
        reset_in => reset_i,
        clk_out  => clk10hz_i
    );
    
    clkTimer_i <= clk100hz_i when switches_in(7) = '1' else clk10hz_i;
    leds_out(7)  <= switches_in(7);
    
-- reset
    reset_proc: process is begin
        wait until rising_edge(sysclk_in);
        if reset_cnt_i = 0 then
            reset_cnt_i <= 0;
            reset_i <= '0';
        else
            reset_cnt_i <= reset_cnt_i - 1;
            reset_i <= '1';
        end if;
    end process;
end Behavioral;

