library IEEE;
    use IEEE.STD_LOGIC_1164.all;

package TIMER_PKG is
-- Global Types
    type TIMER_VALUE_TYPE is record
        min   : integer range 0 to 9;
        sec10 : integer range 0 to 5;
        sec   : integer range 0 to 9;
        tenth : integer range 0 to 9;
    end record;

    type ALU_INSTRUCTION_TYPE is (
        ALU_NONE,               -- do nothing
        ALU_STORE,              -- save current timer value into private register
        ALU_LOAD,               -- load stored value is current timer value
        ALU_DEC_TIMER,          -- decrement timer by a tenth of a second
        ALU_SINGLE_INC_MIN,     -- increment only the minute value. an overflow does not cascade to other digits
        ALU_SINGLE_DEC_MIN,     -- decrement only the minute value. an underflow does not cascade to other digits
        ALU_SINGLE_INC_SEC10,   -- increment only the second tens value. an overflow does not cascade to other digits
        ALU_SINGLE_DEC_SEC10,   -- decrement only the second tens value. an underflow does not cascade to other digits
        ALU_SINGLE_INC_SEC,     -- increment only the second ones value. an overflow does not cascade to other digits
        ALU_SINGLE_DEC_SEC      -- decrement only the second ones value. an underflow does not cascade to other digits
    );
    
-- Components
    component ALU is
        port (
            sysclk_in     : in std_logic;
            reset_in          : in std_logic;

            instruction_in    : in  ALU_INSTRUCTION_TYPE := ALU_NONE;

            timer_out         : out TIMER_VALUE_TYPE;
            timer_is_zero_out : out std_logic
        );
    end component;

    component CLOCK_DIVIDER is
    generic (
         SOURCE_FREQUENCY : positive := 50e6; -- Hz
         TARGET_FREQUENCY : positive :=  1e3  -- Hz
     );
     
     port (
         clk_in    : in std_logic;
         reset_in  : in std_logic;
         clk_out   : out std_logic
     );
    end component;
    
    component DISPLAY is
        generic (
            BRIGHTNESS_BUST_FACTOR : positive := 10; -- No. of multiplexing phases a dark led is turned of compared to a bright one
            DIGITS_NO : positive := 4                -- No. of digits supported by this controller
        );
        
        port (
            clk_in    : in std_logic;
            reset_in  : in std_logic;
            
            data_in       : in std_logic_vector(4 * DIGITS_NO - 1 downto 0);
            brightness_in : in std_logic_vector(DIGITS_NO - 1 downto 0);
            dots_in       : in std_logic_vector(DIGITS_NO - 1 downto 0);
            
            digits_out   : out std_logic_vector(DIGITS_NO - 1 downto 0);
            segments_out : out std_logic_vector(7 downto 0)
        );
    end component;
    
    component DEBOUNCER is
         generic (
              SAMPLING_WIDTH : positive := 8 -- number of sampling cycles the input has to be stable
         );
         
         port (
              sysclk_in       : in STD_LOGIC;
              sampling_clk_in : in STD_LOGIC;
              reset_in  : in STD_LOGIC;
              
              button_in   : in  STD_LOGIC;
              
              button_out  : out STD_LOGIC; -- asserted as long as button is pressed
              released_out : out STD_LOGIC; -- single spike when button is released
              pressed_out : out STD_LOGIC -- single spike when button gets pushed
         );
    end component;
    
    component CONTROLLER is
        port (
            -- Clocks and Reset
            sysclk_in   : in std_logic;
            clk10hz_in  : in std_logic;
            reset_in    : in std_logic;

            -- Alu
            alu_instruction_out : out ALU_INSTRUCTION_TYPE;
            timer_is_zero_in : in std_logic;
        
            -- Key Controls
            keyUp_in    : in std_logic;
            keyDown_in  : in std_logic;
            keyLeft_in  : in std_logic;
            keyRight_in : in std_logic;
            keySet_in   : in std_logic;

            -- 7 Segment Display
            disp_highlight_out : out std_logic_vector(3 downto 0);
            disp_dots_out      : out std_logic_vector(3 downto 0);

            -- Alarm
            alarm_out   : out std_logic
        );
    end component;

end TIMER_PKG;

package body TIMER_PKG is
end TIMER_PKG;