library IEEE;
    use IEEE.STD_LOGIC_1164.ALL;

entity DEBOUNCER is
    generic (
        SAMPLING_WIDTH : positive := 8 -- number of sampling cycles the input has to be stable
    );
    
    port (
        sysclk_in       : in STD_LOGIC;
        sampling_clk_in : in STD_LOGIC;
        reset_in  : in STD_LOGIC;
        
        button_in   : in  STD_LOGIC;
        
        button_out  : out STD_LOGIC;  -- asserted as long as button is pressed
        released_out : out STD_LOGIC; -- single spike when button is released
        pressed_out : out STD_LOGIC   -- single spike when button gets pushed
    );
end DEBOUNCER;

architecture Behavioral of DEBOUNCER is
    signal shift_reg_i : STD_LOGIC_VECTOR(SAMPLING_WIDTH - 1 downto 0) := (others => '0');
    signal stable_high_i : STD_LOGIC := '0';
    signal stable_low_i  : STD_LOGIC := '0';
    
-- TODO: Hier die Zustandsmenge mit zugehrigen Signalen einfügen
type db_states_type is (db_released, db_down, db_pressed, db_up);
signal db_state_i , db_next_state_i : db_states_type;

begin
-- TODO: Hier die zwei FSM Prozesse einfügen

proc_sync: process is begin
		wait until rising_edge(sysclk_in);
		db_state_i <= db_next_state_i;
end process;

proc_comb: process (db_state_i, stable_high_i, stable_low_i, button_in)
is begin

		db_next_state_i <= db_state_i; 	-- implizierte eigenkante

		case db_state_i is
			when db_released => 	if (stable_high_i = '1') then
											db_next_state_i <= db_down;
										end if;
			when db_down => db_next_state_i <= db_pressed;	
			when db_pressed =>
					if (stable_low_i = '1') then
							db_next_state_i <= db_up;
					end if;
			when db_up => db_next_state_i <= db_released;
		end case;
-- Output Function

		button_out <= '0';
		pressed_out <= '0';
		released_out <= '0';
		
		case db_state_i is
			when db_down		=> 		button_out <= '1';
												pressed_out <= '1';
			when db_pressed	=>			button_out <= '1';
			when db_up			=>			released_out <= '1';
			when others			=>			null;
		end case;
		
		
end process;

        
-- sample process synchronous to the sampling clock. it outputs to internal
-- signals (stable_high_i, stable_low_i) indicate the current button state.
    proc_sampling: process is
        variable and_v, or_v : STD_LOGIC;
    begin
        wait until rising_edge(sampling_clk_in);
    
        and_v := '1'; or_v := '0';
        for i in SAMPLING_WIDTH-1 downto 0 loop
            and_v := and_v and shift_reg_i(i);
            or_v  := or_v  or  shift_reg_i(i);
        end loop;

        shift_reg_i <= shift_reg_i(SAMPLING_WIDTH-2 downto 0) & button_in;
        stable_high_i <= and_v;
        stable_low_i  <= not or_v;
    end process;

end Behavioral;