library IEEE;
    use IEEE.STD_LOGIC_1164.ALL;
    use IEEE.NUMERIC_STD.ALL;
    use work.TIMER_PKG.ALL;


entity ALU is
    port (
        sysclk_in     : in std_logic;
        reset_in          : in std_logic;
        
        instruction_in    : in  ALU_INSTRUCTION_TYPE := ALU_NONE;
        
        timer_out         : out TIMER_VALUE_TYPE;
        timer_is_zero_out : out std_logic
    );
end ALU;

architecture Behavior of ALU is
    signal
        timer_i, next_timer_i,
        timer_mem_i, next_timer_mem_i : TIMER_VALUE_TYPE;

    signal
        is_zero_i, next_is_zero_i : std_logic;
begin
    proc_sync: process is begin
        wait until rising_edge(sysclk_in);
        is_zero_i <= next_is_zero_i;
        timer_i   <= next_timer_i;
        timer_mem_i <= next_timer_mem_i;
    end process;

    timer_out <= timer_i;
    timer_is_zero_out <= is_zero_i;

    proc_async: process(sysclk_in, reset_in, instruction_in, timer_i, timer_mem_i, is_zero_i) is begin
        if (timer_i.min = 0 and timer_i.sec10 = 0 and timer_i.sec = 0 and timer_i.tenth = 0) then
            next_is_zero_i <= '1';
        else
            next_is_zero_i <= '0';
        end if;

        next_timer_i     <= timer_i;     -- latch
        next_timer_mem_i <= timer_mem_i; -- latch
        
        case(instruction_in) is
            when ALU_NONE =>
                next_timer_i <= timer_i;
                
            when ALU_LOAD =>
                next_timer_i <= timer_mem_i;
                
            when ALU_STORE =>
                next_timer_mem_i <= timer_i;
                
            when ALU_DEC_TIMER =>
                if timer_i.tenth = 0 then
                    next_timer_i.tenth <= 9;

                    if timer_i.sec = 0 then
                        next_timer_i.sec <= 9;

                        if timer_i.sec10 = 0 then
                            next_timer_i.sec10 <= 5;

                            if timer_i.min = 0 then
                                next_timer_i <= (others => 0);
                                next_is_zero_i <= '1';
                            else
                                next_timer_i.min <= timer_i.min - 1;
                            end if;
                        else
                            next_timer_i.sec10 <= timer_i.sec10 - 1;
                        end if;
                    else
                        next_timer_i.sec <= timer_i.sec - 1;
                    end if;

                else
                    next_timer_i.tenth <= timer_i.tenth - 1;
                end if;

                
            when ALU_SINGLE_INC_MIN =>
                if timer_i.min = 9 then
                    next_timer_i.min <= 0;
                else
                    next_timer_i.min <= timer_i.min + 1;
                end if;

            when ALU_SINGLE_DEC_MIN =>
                if timer_i.min = 0 then
                    next_timer_i.min <= 9;
                else
                    next_timer_i.min<= timer_i.min - 1;
                end if;

            when ALU_SINGLE_INC_SEC10 =>
                if timer_i.sec10 = 5 then
                    next_timer_i.sec10 <= 0;
                else
                    next_timer_i.sec10 <= timer_i.sec10 + 1;
                end if;

            when ALU_SINGLE_DEC_SEC10 =>
                if timer_i.sec10 = 0 then
                    next_timer_i.sec10 <= 5;
                else
                    next_timer_i.sec10 <= timer_i.sec10 - 1;
                end if;

            when ALU_SINGLE_INC_SEC =>
                if timer_i.sec = 9 then
                    next_timer_i.sec <= 0;
                else
                    next_timer_i.sec <= timer_i.sec + 1;
                end if;

            when ALU_SINGLE_DEC_SEC =>
                if timer_i.sec = 0 then
                    next_timer_i.sec <= 9;
                else
                    next_timer_i.sec <= timer_i.sec - 1;
                end if;

        end case;

        if reset_in = '1' then
            next_timer_i <= (min => 1, others => 0);
        end if;
    end process;
end Behavior;