-- Controller for a multiplexed 7-segment multi-digit display

library IEEE;
    use IEEE.STD_LOGIC_1164.ALL;
    use IEEE.NUMERIC_STD.ALL;
    
entity DISPLAY is
   generic (
      BRIGHTNESS_BUST_FACTOR : positive := 10; -- No. of multiplexing phases a dark led is turned of compared to a bright one
      DIGITS_NO : positive := 4                -- No. of digits supported by this controller
   );
   
   port (
      clk_in    : in std_logic;
      reset_in  : in std_logic;
      
      data_in       : in std_logic_vector(4 * DIGITS_NO - 1 downto 0);
      brightness_in : in std_logic_vector(DIGITS_NO - 1 downto 0);
      dots_in       : in std_logic_vector(DIGITS_NO - 1 downto 0);
      
      digits_out   : out std_logic_vector(DIGITS_NO - 1 downto 0);
      segments_out : out std_logic_vector(7 downto 0)
   );
end DISPLAY;

architecture RTL of DISPLAY is
   signal data_shift_reg  : std_logic_vector(data_in'range) := (others => '0');
   signal dots_shift_reg  : std_logic_vector(dots_in'range) := (others => '0');
   signal digit_shift_reg : std_logic_vector(digits_out'range) := (others => '0');

   -- oversampling counter. "dark" digits are turned on, only if counter = 0
   signal brightness_bust_cnt   : integer range 0 to BRIGHTNESS_BUST_FACTOR := 0;
   
begin
   proc_multiplexer: process is
   begin
      wait until rising_edge(clk_in);
   
      if reset_in = '1' or digit_shift_reg(digit_shift_reg'high) = '1' then
         digit_shift_reg <= (0 => '1', others => '0');
         data_shift_reg  <= data_in;
         dots_shift_reg  <= dots_in;

         if reset_in = '1' or BRIGHTNESS_BUST_FACTOR = 0 then
            brightness_bust_cnt <= 0;
         elsif brightness_bust_cnt = 0 then
            brightness_bust_cnt <= BRIGHTNESS_BUST_FACTOR;
         else
            brightness_bust_cnt <= brightness_bust_cnt - 1;
         end if;
         
      else
         digit_shift_reg <= digit_shift_reg(digit_shift_reg'high - 1 downto 0) & "0";
         
         data_shift_reg  <= "0000" & data_shift_reg(data_shift_reg'high downto 4);
         dots_shift_reg  <=    "0" & dots_shift_reg(dots_shift_reg'high downto 1);
      end if;
   end process;
   
   -- select active digit by driving it's common anode
   digits_out    <=
      (others => '1') when reset_in = '1' else
      (not digit_shift_reg) when brightness_bust_cnt = 0 else
      (not (digit_shift_reg and brightness_in));  -- anode "low-active"

   --    --a--
   --   |     |
   --   f     b
   --   |     |
   --    --g--
   --   |     |
   --   e     c
   --   |     |
   --    --d--
   -- The segments are controlled by one cathode per segement yielding a low-active behavior.
   segments_out(6 downto 0) <= 
      "1111111" when reset_in = '1' else
   --  abcdefg
      "1000000" when data_shift_reg(3 downto 0) = X"0" else
      "1111001" when data_shift_reg(3 downto 0) = X"1" else
      "0100100" when data_shift_reg(3 downto 0) = X"2" else
      "0110000" when data_shift_reg(3 downto 0) = X"3" else
      "0011001" when data_shift_reg(3 downto 0) = X"4" else
      "0010010" when data_shift_reg(3 downto 0) = X"5" else
      "0000010" when data_shift_reg(3 downto 0) = X"6" else
      "1111000" when data_shift_reg(3 downto 0) = X"7" else
      "0000000" when data_shift_reg(3 downto 0) = X"8" else
      "0010000" when data_shift_reg(3 downto 0) = X"9" else
      "0001000" when data_shift_reg(3 downto 0) = X"A" else
      "0000011" when data_shift_reg(3 downto 0) = X"B" else
      "1000110" when data_shift_reg(3 downto 0) = X"C" else
      "0100001" when data_shift_reg(3 downto 0) = X"D" else
      "0000110" when data_shift_reg(3 downto 0) = X"E" else
      "0001110";
   
   segments_out(7) <= not (dots_shift_reg(0) or reset_in);
end RTL;