package de.uni_frankfurt.prgpr.phase3.storage;

import java.util.HashMap;
import java.util.Map;

/**
 * Configuration information for clients to use during startup
 * 
 * Data stored in this class can be used e.g. to pass state when calling one client from another client
 * 
 * This class implements the Singleton pattern, meaning that there exists only a single global instance
 * of this object.  To access it, call the static `singleton()' method.
 *  
 * @author creichen
 *
 */
public class ClientConfig {
	private Map<String, String> config = new HashMap<>();
	
	/**
	 * Creates a new ClientConfig object.
	 * 
	 * This constructor should only be used for testing; for setting configiration,
	 * use `singleton()', instead.
	 */
	public ClientConfig() {
	}
	
	private static final ClientConfig CONFIG = new ClientConfig();
	
	/**
	 * Retrieves the only ClientConfig object.
	 * 
	 * @return singleton The clientConfig
	 */
	public ClientConfig singleton() {
		return CONFIG;
	}
	
	/**
	 * Gets a configuration parameter
	 * 
	 * @param value Name of the configuration parameter to read
	 * @param defaultValue Default string to return if the configuration parameter wasn't set
	 * @return defaultValue if `value' wasn't set via `put', otherwise the most recently configured value
	 */
	public String get(String value, String defaultValue) {
		if (config.containsKey(value)) {
			return config.get(value);
		}
		return defaultValue;
	}
	
	/**
	 * Configures a configuration parameter
	 * 
	 * @param key Name of the configuration parameter to set
	 * @param value Value to set the configuration parameter to
	 */
	public void put(String key, String value) {
		config.put(key, value);
	}
	
	/**
	 * Remotes a configuration parameter
	 * 
	 * @param key Name of the configuration parameter to remove
	 */
	public void delete(String key) {
		config.remove(key);
	}
	
	
}
