package de.uni_frankfurt.prgpr.phase3.images;

import java.awt.Image;
import java.awt.image.BufferedImage;

/**
 * A set of tiles (numbered in the range[0 ... size())) that are stored in a single image.
 * 
 * @author creichen
 *
 */
public class Tileset {

	protected BufferedImage image;
	protected int celWidth;
	protected int celHeight;
	protected int celColumns;
	protected int celRows;

	/**
	 * Creates a fresh tileset for the given image
	 * 
	 * The tileset must be initialised by calling setTileDimensions().
	 * 
	 * @param image The image containing the graphics for the tileset
	 */
	public Tileset(BufferedImage image) {
		this.image = image;
	}

	/**
	 * Sets the sizes of the individual tiles stored in the tilesheet
	 * 
	 * Used during initialisation
	 *  
	 * @param width The width of each cel
	 * @param height The height of each cel
	 */
	protected void setTileDimensions(int width, int height) {
		this.celWidth = width;
		this.celHeight = height;
		this.celColumns = this.image.getWidth() / width;
		this.celRows = this.image.getHeight() / height;
	}
	
	/**
	 * Determines the number of tiles that this tileset possesses
	 * 
	 * @return The number of tiles stored in this tileset
	 */
	public int size() {
		return this.celColumns * this.celRows;
	}

	/**
	 * Retrieves a particular tile image
	 * 
	 * @param tileNr Number of the tile image
	 * @return The tile image, or null if tileNr is invalid
	 */
	public Image getTile(int tileNr) {
		if (tileNr < 0 || tileNr >= celColumns * celRows) {
			return null;
		}
		int xgrid = tileNr % celColumns;
		int ygrid = tileNr / celColumns;
		return image.getSubimage(celWidth * xgrid, celHeight * ygrid, celWidth, celHeight);
	}

	/**
	 * Prints useful information about the spritesheet in string form
	 */
	@Override
	public String toString() {
		return "TileSheet(" + this.image.getWidth() + "x" + this.image.getHeight() + ", celSize=" + celWidth + "x" + celHeight + ", celGrid = " + celColumns + "x" + celRows + ")";
	}
}