package de.uni_frankfurt.prgpr.phase3.images;

import java.awt.Image;
import java.awt.image.BufferedImage;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Spritesheets represent a set of `cels', or animation frames, together with `actions' that describe
 * animations constructed from the cels.
 * 
 * Spritesheets may also be dyeable, meaning that parts of the spritesheet may have particular colours
 * swapped out by a colour gradient.  This allows e.g. a white `shirt' spritesheet to be re-used as a red,
 * green, blue, or purple spritesheet.
 * 
 * @author creichen
 *
 */
public class Spritesheet extends Tileset {
	EnumSet<DyeChannel> dyeChannels;
	private Map<String, Map<Direction, Animation>> animations = new HashMap<>(); // all animations
	
	/**
	 * Constructs a new spritesheet
	 * 
	 * @param img The image that this spritesheet is based on
	 * @param dyes The number of dye channels supported by the spritesheet, for dyeing parts of the image
	 */
	public Spritesheet(BufferedImage img, EnumSet<DyeChannel> dyes) {
		super(img);
		this.dyeChannels = dyes;
		this.setCelDimensions(img.getWidth(), img.getHeight());
	}
	
	/**
	 * Sets the sizes of the individual cels stored in the spritesheet
	 * 
	 * Used during initialisation
	 *  
	 * @param width The width of each cel
	 * @param height The height of each cel
	 */
	protected void setCelDimensions(int width, int height) {
		super.setTileDimensions(width, height);
	}
	/**
	 * Retrieves a particular cel image
	 * 
	 * @param celNr Number of the cel image
	 * @return The cel image, or null if celNr is invalid
	 */
	public Image getCel(int celNr) {
		return super.getTile(celNr);
	}

	
	/**
	 * Checks whether the spritesheet can be dyed
	 * 
	 * @return true iff the spritesheet can be dyed
	 */
	public boolean
	isDyeable() {
		return !this.dyeChannels.isEmpty();
	}
	
	/**
	 * Returns the set of all dyes that we can apply to this spritesheet
	 * 
	 * @return A set of all DyeChannels that we can set for this spritesheet
	 */
	public Set<? extends DyeChannel>
	getDyes() {
		return this.dyeChannels;
	}

	/**
	 * Adds a single animation to the spritesheet
	 * 
	 * @param name Name of the animation to add
	 * @param direction Direction within that animation
	 * @param animation The animation object describing the animation
	 */
	void addAnimation(String name, Direction direction,	Animation animation) {
		Map<Direction, Animation> animationRecord;
		if (!this.animations.containsKey(name)) {
			animationRecord = new HashMap<>();
			this.animations.put(name, animationRecord);
		} else {
			animationRecord = this.animations.get(name);
		}
		animationRecord.put(direction, animation);
	}
	
	/**
	 * Lists all actions supported by this spritesheet
	 * 
	 * @return The set of all actions that this spritesheet can animate
	 */
	public Set<? extends String> getActions() {
		return this.animations.keySet();
	}
	
	/**
	 * Lists all directions for the given action
	 * 
	 * @param action The action whose animations we should check
	 * @return A set of all supported directions 
	 */
	public Set<? extends Direction> getDirections(String action) {
		if (this.animations.containsKey(action)) {
			return this.animations.get(action).keySet();
		}
		return Collections.emptySet();
	}
	

	/**
	 * Constructs a fresh spritesheet obtained from this one by dyeing some or all of the colour channels
	 *  
	 * @param dyeSpecs Any number of Dyeing specifications (channel, gradient)
	 * @return A new spritesheet in which colours of `channel' have been replaced by their respective `gradient' colours 
	 */
	public Spritesheet dye(DyeSpec ... dyeSpecs) {
		Gradient[] dyes = new Gradient[8];
		for (DyeSpec dyespec : dyeSpecs) {
			dyes[dyespec.getChannel().getColorIndex()] = dyespec.getGradient();
		}
		
		Spritesheet sheet = (Spritesheet) this.clone();
		BufferedImage sourceImage = sheet.image;
		BufferedImage targetImage = new BufferedImage(this.image.getWidth(), this.image.getHeight(), BufferedImage.TYPE_INT_ARGB);
		sheet.image = targetImage;
		
		for (int y = 0; y < sourceImage.getHeight(); y++) {
			for (int x = 0; x < sourceImage.getWidth(); x++) {
				int rgb = sourceImage.getRGB(x, y); // returned as four bytes (ABGR) packed into one 32 bit int
				int alpha = rgb >>> 24;
				if (alpha == 0x00) { // optimisation: skip transparent bytes
					targetImage.setRGB(x, y, rgb); // no need to dye
					continue;
				}
				
				int dyeChannelID = 0; // dye channel ID, or 0 if not dyed
				int intensity = 0; // the dye intensity (`w' value)
				int rgbAnalysis = rgb;
				
				for (int i = 1; i < 8; i <<= 1) {
					int localIntensity = rgbAnalysis & 0xff;
					if (intensity == 0) {
						intensity = localIntensity;
					}
					if (localIntensity != 0) {
						if (localIntensity != intensity) {
							dyeChannelID = 0; // mismatch
							break;
						}
						dyeChannelID |= i;
					}
					rgbAnalysis >>= 8;
				}
				if (dyeChannelID != 0 && dyes[dyeChannelID] != null) {
					Gradient gradient = dyes[dyeChannelID];
					rgb = gradient.getColorRGB(intensity);
					rgb |= alpha << 24;
				}
				targetImage.setRGB(x, y, rgb);
			}
		}
		System.err.println();

		return sheet;
	}
	
	/**
	 * Clones (i.e., copies) the current spritesheet, including contents
	 * 
	 * @return  A clone of the spritesheet
	 */
	@Override
	public Spritesheet clone() {
		Spritesheet retval = new Spritesheet(this.image, this.dyeChannels);
		retval.animations = this.animations;
		retval.setCelDimensions(celWidth, celHeight);
		return retval; 
	}
	
	/**
	 * Starts an animation of the given action and direction
	 * 
	 * @param action The action to animate
	 * @param direction The direction to animate in
	 * @return A state object that allows us to determine the cel numbers and the delays between cel times
	 */
	public AnimationState startAction(String action, Direction direction) {
		if (!this.animations.containsKey(action)) {
			return null;
		}
		Map<Direction, Animation> directions = this.animations.get(action);
		if (!directions.containsKey(direction)) {
			direction = Direction.DEFAULT;
		}
		if (!directions.containsKey(direction)) {
			return null;
		}
		return directions.get(direction).newState();
	}
	/**
	 * Prints useful information about the spritesheet in string form
	 */
	@Override
	public String toString() {
		return "Spritesheet(" + this.image.getWidth() + "x" + this.image.getHeight() + ", celSize=" + celWidth + "x" + celHeight + ", celGrid = " + celColumns + "x" + celRows + ", dyeChannels=" + this.dyeChannels + ", actions=" + getActions() + ")";
	}
}
