package de.uni_frankfurt.prgpr.phase3.images;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;

import javax.imageio.ImageIO;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Class for loading spritesheets and tilesets from a zip file
 * 
 * @author creichen
 *
 */
public class ImageLoader {
	
	private String filename;

	private ImagesetRepository<Spritesheet> spritesheets = null;
	private ImagesetRepository<Tileset> tilesets = null;
	private ImagesetRepository<BufferedImage> items = null;

	// XML parser
	private DocumentBuilder xmlParser = null;

	private ZipFile zipFile;
	
	/**
	 * Creates a spritesheet and tileset loader for the specified file name
	 * 
	 * @param filename Name of the zip file to load
	 */
	public ImageLoader(String filename) {
		this.filename = filename;
		try {
			this.xmlParser = DocumentBuilderFactory.newInstance().newDocumentBuilder();
		} catch (ParserConfigurationException exn) {
			throw new RuntimeException(exn);
		}
	}

	/**
	 * Debug method to print XML documents
	 * 
	 * @param doc The XML document to print
	 * @param out The output stream to print to
	 */
	@SuppressWarnings("unused")
	private static void printDocument(Document doc, OutputStream out) {
		// code by http://stackoverflow.com/users/203907/bozho
		try {
			TransformerFactory tf = TransformerFactory.newInstance();
			Transformer transformer = tf.newTransformer();
			transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "no");
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
			transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");

			transformer.transform(new DOMSource(doc), 
					new StreamResult(new OutputStreamWriter(out, "UTF-8")));
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Initialises the ImageLoader by indexing all viable spritesheets and tilesets and preparing them for loading
	 * @throws IOException if the file could not be loaded
	 */
	public void init() throws IOException {
		if (spritesheets != null) {
			return;
		}
		
		spritesheets = new ImagesetRepository<Spritesheet>(".xml", new ParserBuilder<Spritesheet>() {
			public ImageParser<Spritesheet>
			newParser() {
				return new SpritesheetParser();
			}
		});
		tilesets = new ImagesetRepository<Tileset>(".tsx", new ParserBuilder<Tileset>() {
			public ImageParser<Tileset>
			newParser() {
				return new TilesetParser();
			}
		});
		items = new ImagesetRepository<BufferedImage>(".png", new ParserBuilder<BufferedImage>() {
			public ImageParser<BufferedImage>
			newParser() {
				return new ImageParser<BufferedImage>() {
					@Override
					public BufferedImage parse(ZipEntry zipentry) {
						return loadImage(zipentry.getName());
					}
				};
			}
		}) {
			@Override
			protected boolean filenameIsInteresting(String filename) {
				return super.filenameIsInteresting(filename) && filename.startsWith("graphics/items/");
			}
		};
		
		this.zipFile = new ZipFile(this.filename);
		Enumeration<? extends ZipEntry> zipIterator = zipFile.entries();
		while (zipIterator.hasMoreElements()) {
			ZipEntry z = zipIterator.nextElement();
			tilesets.tryAdd(z);
			spritesheets.tryAdd(z);
			items.tryAdd(z);
		}
	}
	
	/**
	 * Retrieves a spritesheet by name
	 * 
	 * @param name Name of the spritesheet
	 * @return A spritesheet, obtained from the given resource, or null on error
	 */
	public Spritesheet getSpritesheet(String name) {
		return this.spritesheets.get(name);
	}

	/**
	 * Retrieves a tileset by name
	 * 
	 * @param name Name of the tileset
	 * @return A tileset, obtained from the given resource, or null on error
	 */
	public Tileset getTileset(String name) {
		return this.tilesets.get(name);
	}

	/**
	 * Retrieves an item by name
	 * 
	 * @param name Name of the item to load
	 * @return An image for the given item name, or null on error
	 */
	public BufferedImage getItem(String name) {
		return this.items.get(name);
	}

	/**
	 * Helper function to extract the bytes for a given ZipEntry
	 * 
	 * @param entry ZipEntry to extract the data for
	 * @return The input stream containing the given zip entry contents
	 */
	private InputStream getData(ZipEntry entry) {
		try {
			InputStream s = zipFile.getInputStream(entry);
			return s;
			
		} catch (IOException exn) {
			throw new RuntimeException(exn);
		}
	}
	
	/**
	 * Helper function to extract the bytes for a given ZipEntry
	 * 
	 * @param entry ZipEntry to extract the data for
	 * @return The input stream containing the given zip entry contents
	 */
	private InputStream getDataForFilename(String data) {
		try {
			ZipEntry entry = zipFile.getEntry(data);
			if (entry == null) {
				throw new RuntimeException("Missing file entry: " + data);
			}
			InputStream s = zipFile.getInputStream(entry);
			return s;
			
		} catch (IOException exn) {
			throw new RuntimeException(exn);
		}
	}

	/**
	 * Retrieves the set of all spritesheet names
	 * 
	 * @return A set of all spritesheet names
	 */
	public Set<? extends String> getSpritesheetNames() {
		return this.spritesheets.getNames();
	}
	
	/**
	 * Retrieves the set of all tileset names
	 * 
	 * @return A set of all tileset names
	 */
	public Set<? extends String> getTilesetNames() {
		return this.tilesets.getNames();
	}
	
	/**
	 * Retrieves the set of all item names
	 * 
	 * @return A set of all item names
	 */
	public Set<? extends String> getItemNames() {
		return this.items.getNames();
	}
	
	private static ImageLoader DEFAULT = null;
	
	/**
	 * Loads sprites from the default location
	 * @return A spritesheet loader from the default location
	 * @throws IOException
	 */
	public static ImageLoader defaultLoader() throws IOException {
		if (DEFAULT != null) {
			return DEFAULT;
		}
		DEFAULT = new ImageLoader(System.getProperty("user.home") + File.separator + "prgpr" + File.separator + "graphics.zip");
		DEFAULT.init();
		return DEFAULT;
	}
	

	public static void main(String[] args) throws IOException {
		ImageLoader loader = defaultLoader();
		System.out.println(loader.getSpritesheetNames());
		System.out.println(loader.getSpritesheet("graphics/sprites/model/female"));
		System.out.println(loader.getTilesetNames());
		System.out.println(loader.getTileset("tilesets/desert1"));
		System.out.println(loader.getItemNames());
	}

	/**
	 * A class that constructs fresh parsers for XML files for ImagesetTypes
	 * @author creichen
	 *
	 * @param <ImagesetType> The particular ImagesetType (Tileset or Spritesheet) to parse
	 */
	private static interface ParserBuilder<ImagesetType> {
		/**
		 * Constructs a parser for XML files to ImagesetType objects
		 * @return A freshly allocated parser
		 */
		public ImageParser<ImagesetType> newParser();
	}
	
	/**
	 * Parses an XML file into an ImagesetType
	 * @author creichen
	 *
	 * @param <ImagesetType> The particular ImagesetType (Tileset or Spritesheet) to parse
	 */
	private abstract class ImageParser<ImagesetType> {
		/**
		 * Tries to parse an ImagesetType out of the specified zip file entry
		 * 
		 * @param ZipEntry The zip file entry to parse from
		 * @return A parsed ImagesetType, or <tt>null</tt> on error
		 */
		public abstract ImagesetType parse(ZipEntry zipentry);
		
		/**
		 * Parses an XML input stream into an XML document element
		 * 
		 * @param indoc
		 * @return
		 */
		public Element parseDoc(InputStream indoc) {
			Document doc;
			try {
				doc = xmlParser.parse(indoc);
			} catch (SAXException e) {
				e.printStackTrace();
				return null;
			} catch (IOException e) {
				e.printStackTrace();
				return null;
			}
			return doc.getDocumentElement();
		}
		
		/**
		 * Obtains an integer-valued attribute from an XML element
		 * 
		 * @param elt The element to investigate
		 * @param name The attribute to investigate
		 * @return The attribute's integer value, or 0 if the attribute is missing
		 */
		int intAttribute(Element elt, String name) {
			if (elt.hasAttribute(name)) {
				return Integer.parseInt(elt.getAttribute(name));
			}
			return 0;
		}
		
		/**
		 * Loads an image file, given its filename
		 * 
		 * @param filename Name of the image file to load
		 * @return The resultant BufferedImage; if not found, raises a RuntimeException
		 */
		BufferedImage loadImage(String filename) {
			// turn into absolute path
			while (filename.startsWith("../")) {
				filename = filename.substring(3);
			}
			BufferedImage bufferedImage = null;
			try {
				bufferedImage = ImageIO.read(getDataForFilename(filename));
			} catch (IOException e) {
				e.printStackTrace();
				throw new RuntimeException(e);
			}
			return bufferedImage;
		}
	}

	/**
	 * A repository for particular kinds of images (Spritesheets, Tilesets) with the ability to load them on demand
	 * 
	 * Also provides functionality to select relevant data from ZipEntries.
	 * 
	 * @author creichen
	 *
	 * @param <ImagesetType> The type of image to store
	 */
	private class ImagesetRepository<ImagesetType> {
		private Map<String, ZipEntry> imageMap = new HashMap<>();
		private Map<String, ImagesetType> imageCache = new HashMap<>();
		private ParserBuilder<ImagesetType> parserBuilder;
		private String fileSuffix;
		
		/**
		 * Creates a fresh ImagesetRepository.
		 * 
		 * @param expectedSuffix Filename suffix for files that this repository expects to handle
		 * @param parserBuilder A ParserBuilder for parsing files with the specified filename suffix
		 */
		public ImagesetRepository(String expectedSuffix, ParserBuilder<ImagesetType> parserBuilder) {
			this.fileSuffix = expectedSuffix;
			this.parserBuilder = parserBuilder;
		}
		
		/**
		 * Determines whether a filename is of interest to us
		 * 
		 * @param filename
		 * @return true iff the filename is a file name that we should apply the parser to
		 */
		protected boolean filenameIsInteresting(String filename) {
			return filename.endsWith(fileSuffix);
		}

		/**
		 * Attempts to add a file to the repository.
		 * 
		 * @param entry ZipEntry to attempt to add.  If the entry's file name matches the given suffix, the repository
		 * records the entry for later retrieval.
		 */
		public void tryAdd(ZipEntry entry) {
			String filename = entry.getName();
			if (filenameIsInteresting(filename)) {
				String shortName = filename.substring(0, filename.length() - fileSuffix.length());
				this.imageMap.put(shortName, entry);
			}
		}
		
		/**
		 * Retrieves a set of all entries that tryAdd() successfully added
		 * 
		 * @return A set of all names of all images stored in this repository
		 */
		public Set <? extends String> getNames() {
			return imageMap.keySet();
		}
		
		/**
		 * Retrieves a given image by name
		 * 
		 * @param name Name of the image to look up
		 * @return The corresponding image
		 */
		public ImagesetType get(String name) {
			if (imageCache.containsKey(name)) {
				return imageCache.get(name);
			}
			ZipEntry entry = imageMap.get(name);
			if (entry == null) {
				return null;
			}
			ImageParser<ImagesetType> parser = parserBuilder.newParser();
			ImagesetType imageset = parser.parse(entry);
			if (imageset != null) {
				imageCache.put(name, imageset);
			}
			return imageset;
		}
	}

	/**
	 * Class for parsing a single Tileset
	 * @author creichen
	 *
	 */
	private class TilesetParser extends ImageParser<Tileset> {
		/**
		 * Tries to parse a tileset out of the specified zip file entry
		 * 
		 * @param zipentry The zip file entry to read from
		 * @return A parsed tileset , or <tt>null</tt> on error
		 */
		@Override
		public Tileset parse(ZipEntry zipentry) {
			Element elt = parseDoc(getData(zipentry));
			int tilewidth = intAttribute(elt,  "tilewidth"); 
			int tileheight = intAttribute(elt,  "tileheight");
			NodeList nl = elt.getChildNodes();
			
			for (int i = 0; i < nl.getLength(); i++) {
				Node n = nl.item(i);
				if (n instanceof Element) {
					Element child = (Element)n;
				
					if (child.getTagName().equals("image")) {
						Tileset tileset = new Tileset(loadImage(child.getAttribute("source")));
						tileset.setTileDimensions(tilewidth, tileheight);
						
						return tileset;
					}
				}
			}
			return null;
		}
	}
	
	/**
	 * Class for parsing a single spritesheet
	 * @author creichen
	 *
	 */
	private class SpritesheetParser extends ImageParser<Spritesheet> {
		private Spritesheet spritesheet;
		private int width;
		private int height;

		/**
		 * Retrieves the spritesheet built up by the spritesheet parser
		 * 
		 * @return The generated spritesheet
		 */
		public Spritesheet getSpritesheet() {
			return spritesheet;
		}
		
		/**
		 * Tries to parse a spritesheet out of the specified zip file entry
		 * 
		 * @param istream The zip file entry to parse from
		 * @return A parsed spritesheet, or <tt>null</tt> on error
		 */
		@Override
		public Spritesheet parse(ZipEntry zipentry) {
			parseToplevel(getData(zipentry));
			return getSpritesheet();
		}
		
		/**
		 * Checks if the element is an '&lt;include file="..."&gt;' element; if so, includes the specified file 
		 * @param elt The element to check
		 * @return The document element of the included specification, or null
		 */
		public Element includeIfNeeded(Element elt) {
			if (elt.getTagName().equals("include")) {
				return parseDoc(getDataForFilename(elt.getAttribute("file")));
			}
			return null;
		}
		
		/**
		 * Parses an &lt;imageset&gt; element
		 * 
		 * @param elt The element to parse
		 */
		private void parseImageSet(Element elt) {
			if (spritesheet != null) { // already have a spritesheet; obey override semantics
				return;
			}
			String filename = elt.getAttribute("src");
			EnumSet<DyeChannel> dyeChannels = EnumSet.noneOf(DyeChannel.class);
			int splitIndex;
			if ((splitIndex = filename.indexOf('|')) > 0) {
				int index = splitIndex + 1;
				for (; index < filename.length(); index++) {
					switch (filename.charAt(index)) {
					case 'R': dyeChannels.add(DyeChannel.RED); break;
					case 'G': dyeChannels.add(DyeChannel.GREEN); break;
					case 'B': dyeChannels.add(DyeChannel.BLUE); break;
					case 'C': dyeChannels.add(DyeChannel.CYAN); break;
					case 'M': dyeChannels.add(DyeChannel.MAGENTA); break;
					case 'Y': dyeChannels.add(DyeChannel.YELLOW); break;
					case 'W': dyeChannels.add(DyeChannel.WHITE); break;
					}
				}
				filename = filename.substring(0, splitIndex); 
			}
			BufferedImage bufferedImage = loadImage(filename);
			spritesheet = new Spritesheet(bufferedImage, dyeChannels);
			if (elt.hasAttribute("width") && elt.hasAttribute("height")) {
				width = Integer.parseInt(elt.getAttribute("width"));
				height = Integer.parseInt(elt.getAttribute("height"));
				spritesheet.setCelDimensions(width, height);
			}
		}

		/**
		 * Parses an &lt;animation&gt; tag, which usually represents one direction of animation within a specific animation name
		 * 
		 * @param elt The animation to parse
		 * @param name Name of the animation
		 */
		private void parseAnimation(Element elt, String name) {
			Direction direction = Direction.DEFAULT;
			switch (elt.getAttribute("direction")) {
			case "up": direction = Direction.UP; break;
			case "down": direction = Direction.DOWN; break;
			case "left": direction = Direction.LEFT; break;
			case "right": direction = Direction.RIGHT; break;
			}
			
			Animation animation = new Animation();
			boolean looping = true;
			
			NodeList nl = elt.getChildNodes();
			
			for (int i = 0; i < nl.getLength(); i++) {
				Node n = nl.item(i);
				if (n instanceof Element) {
					Element animationStep = (Element)n;
					
					int xoffset = intAttribute(animationStep, "offsetX");
					int yoffset = intAttribute(animationStep, "offsetY");
					xoffset -= width / 2;
					yoffset -= height;
					
					switch (animationStep.getTagName()) {
					case "frame":
						int index = intAttribute(animationStep, "index");
						animation.addCelSequence(new CelSequence(
								index, index, intAttribute(animationStep, "delay")).setOffsets(xoffset,  yoffset));
						break;
					case "sequence":
						animation.addCelSequence(new CelSequence(
								intAttribute(animationStep, "start"),
								intAttribute(animationStep, "end"),
								intAttribute(animationStep, "delay")).setOffsets(xoffset, yoffset));
						break;
					case "end":
						looping = false;
						break;
					}
				}
			}
			
			animation.setIsLooping(looping);
			spritesheet.addAnimation(name, direction, animation);
		}
		
		/**
		 * Parses an &lt;action&gt; tag
		 * 
		 * @param elt The element to parse
		 */
		private void parseAction(Element elt) {
			String actionName = elt.getAttribute("name");
			
			NodeList nl = elt.getChildNodes();
			
			for (int i = 0; i < nl.getLength(); i++) {
				Node n = nl.item(i);
				if (n instanceof Element) {
					Element actionPart = (Element)n;
					switch (actionPart.getTagName()) {
					case "animation": parseAnimation(actionPart, actionName); break;
					}
				}
			}
		}
			
		/**
		 * Parses a toplevel XML document input stream for a spritesheet specification
		 * 
		 * @param docStream The stream to parse from
		 */
		public void parseToplevel(InputStream docStream) {
			parseToplevelElement(parseDoc(docStream));
		}

		/**
		 * Parses a toplevel spritesheet definition XML document
		 * 
		 * @param body The document element for the spritesheet
		 */
		public void parseToplevelElement(Element body) {
			if (body == null) {
				return;
			}
			
			if (!body.getTagName().equals("sprite")) {
				System.err.println("Not a <sprite>");
				return;
			}
			NodeList nl = body.getChildNodes();
			for (int i = 0; i < nl.getLength(); i++) { 
				Node n = nl.item(i);
				if (n instanceof Element) {
					Element elt = (Element) n;
					switch (elt.getTagName()) {
					case "imageset":
						parseImageSet(elt);
						break;
					case "action":
						parseAction(elt);
						break;
					case "include":
						parseToplevelElement(includeIfNeeded(elt));
					}
				}
			}
		}
	}
}
