package de.uni_frankfurt.prgpr.phase3.images;

import java.awt.Color;
import java.io.Serializable;

/**
 * A gradient from black (zero) to any sequence of colours.
 * 
 * The sequence of colours is specified in the constructor.
 * 
 * @author creichen
 *
 */
public final class Gradient implements Serializable {
	private static final long serialVersionUID = 1L;
	private Color[] colors; // colours in the colour space
	private int colorSpaceSize;

	/**
	 * Defines a colour gradient.
	 * 
	 * @param gradientColors All the colours that should make up the gradient, from lowest to highest intensity.  Black is implicitly included at zero.
	 */
	public Gradient(Color ... gradientColors) {
		this.colors = new Color[gradientColors.length + 1];
		this.colors[0] = Color.BLACK;
		System.arraycopy(gradientColors, 0, this.colors, 1, gradientColors.length);
		if (gradientColors.length == 0) {
			this.colors = new Color[] { Color.BLACK, Color.BLACK };
		}
		recompute();
	}

	/**
	 * Prepares numbers for colour computation that change only when the gradient changes.
	 */
	private void recompute() {
		colorSpaceSize = 256 / (colors.length - 1);
	}
	
	/**
	 * Retrieves the colour of the at the specified intensity.
	 *  
	 * @param intensity The offset in the colour space, must be in [0, 255]
	 * @return A Color object, matching the specified intensity
	 */
	public Color getColorAt(int intensity) {
		int rgb = getColorRGB(intensity);
		return new Color((rgb >>> 16) & 0xff, (rgb >>> 8) & 0xff, rgb & 0xff);
	}
	
	/**
	 * Retrieves gradient colour of the specified intensity.
	 * 
	 * @param intensity The offset in the colour space, must be in [0, 255]
	 * @return An RGB value (without alpha) encoded in three bytes, with B as least significant
	 */
	public int getColorRGB(int intensity) {
		int retval = 0;
		int leftColorIndex = intensity / colorSpaceSize;
		int rightColorIndex = leftColorIndex + 1;
		if (rightColorIndex + 1 >= colors.length) {
			rightColorIndex = colors.length - 1;
		}
		// compute subgradient: from left to right
		int left = colors[leftColorIndex].getRGB();
		int right = colors[rightColorIndex].getRGB();
		int colorSpaceIndex = intensity - (leftColorIndex * colorSpaceSize);
		// Within the subgradient (left, right) we are at position colorSpaceIndex of colorSpaceSize
		
		// precompute division:
		int indexFactor = (0x10000 * colorSpaceIndex) / colorSpaceSize;
		
		for (int i = 0; i < 3; i++) {
			int leftIntensity = left & 0xff;
			int rightIntensity = right & 0xff;
			int intensityDelta = rightIntensity - leftIntensity;
			int componentIntensity = leftIntensity + ((intensityDelta * indexFactor) >> 16); // ">> 16" is equivalent to "/ 0x10000"
			retval |= componentIntensity << (8 * i);
			left >>= 8;
			right >>= 8;
		}
		return retval;
	}
	
	/**
	 * Visualises the contents of this colour gradient class as a string.
	 */
	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer("[");
		for (Color c : this.colors) {
			sb.append(c);
		}
		sb.append("]");
		return sb.toString();
	}
	
	/**
	 * Compares two gradients for equality.
	 */
	@Override
	public boolean equals(Object obj) {
		if (obj == this) {
			return true;
		} else if (obj == null) {
			return false;
		} else if (obj instanceof Gradient) {
			Gradient other = (Gradient) obj;
			if (other.colors.length != this.colors.length) {
				return false;
			}
			for (int i = 0; i < this.colors.length; i++) {
				if (!other.colors[i].equals(this.colors[i])) {
					return false;
				}
			}
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * Computes the hash code for this colour gradient class (for use in hash maps etc.)
	 */
	@Override
	public int hashCode() {
		int hc = 42;
		for (Color c : this.colors) {
			hc <<= 3;
			hc += c.hashCode();
		}
		return hc;
	}
}
