package de.uni_frankfurt.prgpr.phase3.images;

import java.awt.Graphics;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

/**
 * A layered set of sprites, together with code that maintains animation status for the sprites. 
 * 
 * @author creichen
 *
 */
public class Entity implements Serializable {
	private static final long serialVersionUID = 33L;
	private List<SpriteLayer> layers = new ArrayList<>();
	private String defaultAction, lastAction;
	private Direction direction = Direction.DOWN;
	
	/**
	 * Creates a new entity, facing DOWN
	 * 
	 * @param defaultAction The action that the entity should perform initially, and whenever a (non-looping) animation finishes looping 
	 * @param baseSprite The base image to add to sprite layer zero
	 * @param dyes The dyes to apply to the base image
	 */
	public Entity(String defaultAction, String baseSprite, DyeSpec ... dyes) {
		layers.add(new SpriteLayer(baseSprite, dyes));
		this.defaultAction = defaultAction;
		lastAction = defaultAction;
	}

	/**
	 * Adds a sprite layer
	 * 
	 * @param sprite The image to add to the sprite layer
	 * @param dyes The dyes to apply to the image
	 */
	public void addLayer(String sprite, DyeSpec ... dyes) {
		layers.add(new SpriteLayer(sprite, dyes));
		restartAnimation(lastAction);
	}
	
	/**
	 * Modifies an existing sprite layer
	 * 
	 * @param sprite The image to set as that sprite layer
	 * @param dyes The dyes to apply to that image
	 */
	public void setLayer(int layerNr, String sprite, DyeSpec ... dyes) {
		layers.set(layerNr,  new SpriteLayer(sprite, dyes));
		restartAnimation(lastAction);
	}

	/**
	 * Removes an existing sprite layer
	 * 
	 * @param layerNr The layer to remove
	 */
	public void removeLayer(int layerNr) {
		layers.remove(layerNr);
		restartAnimation(lastAction);
	}
	
	/**
	 * Starts an animation
	 * 
	 * @param action The action to animate
	 */
	public void startAction(String action) {
		lastAction = action;
		restartAnimation(action);
	}
	
	/**
	 * Changes the entity's direction
	 * 
	 * @param dir The direction for the entity to face
	 */
	public void setDirection(Direction dir) {
		if (direction != dir) {
			direction = dir;
			restartAnimation(lastAction);
		}
	}
	
	/**
	 * Advances the entity's animation state by the specified number of milliseconds
	 * 
	 * @param milliseconds The time to advance the animation by
	 */
	public void advanceAnimation(int milliseconds) {
		for (SpriteLayer layer : layers) {
			layer.advanceAnimation(milliseconds);
		}
	}
	
	/**
	 * Determines the number of layers in the entity
	 * 
	 * @return The number of entity layers
	 */
	public int size() {
		return layers.size();
	}
	
	/**
	 * Draws the entity
	 * 
	 * @param g The graphics object to draw to
	 * @param x The x coordinate to draw to
	 * @param y The y coordinate to draw to
	 */
	public void draw(Graphics g, int x, int y) {
		for (SpriteLayer layer : layers) {
			layer.draw(g, x, y);
		}
	}
	
	/**
	 * Internally resets the animation cycle
	 * 
	 * @param action The action to reset the animation with
	 */
	private void restartAnimation(String action) {
		for (SpriteLayer layer : layers) {
			layer.startAction(action);
		}
	}
			
	/**
	 * Internal helper class to represent a singular sprite layer
	 * 
	 * @author creichen
	 *
	 */
	private final class SpriteLayer implements Serializable {
		private static final long serialVersionUID = 1L;
		private String name;
		private DyeSpec[] dyeArray;
		private int millisecondsUntilChange;
		private AnimationState animationState;

		/**		
		 * Constructs a fresh sprite layer
		 * 
		 * @param spriteName Name of the spritesheet to use
		 * @param dyes Dyes to apply to the spritesheet
		 */
		public SpriteLayer(String spriteName, DyeSpec[] dyes) {
			this.name = spriteName;
			this.dyeArray = dyes;
		}
		
		/**
		 * Draws the sprite layer
		 * 
		 * @param g The graphics object to draw into
		 * @param x The x coordinate to draw to
		 * @param y The y coordinate to draw to
		 */
		public void draw(Graphics g, int x, int y) {
			g.drawImage(getSpritesheet().getCel(animationState.getCel()),
					animationState.getXOffset() + x,
					animationState.getYOffset() + y,
					null);
		}
		
		/**
		 * Advances the layer's animation state by the specified number of milliseconds
		 * 
		 * @param milliseconds The time to advance the animation by
		 */
		public void advanceAnimation(int milliseconds) {
			millisecondsUntilChange -= milliseconds;
			while (millisecondsUntilChange < 0) {
				if (animationState.getDelay() == 0) {
					millisecondsUntilChange = 0;
					break; // infinite length
				}
				if (animationState.next()) {
					millisecondsUntilChange += animationState.getDelay();
				} else {
					restartAnimation(defaultAction);
					// revert to defaultAction as soon as the first layer runs out of things to do
				}
			}
		}
		
		/**
		 * Starts an action (animation) in the layer
		 * 
		 * @param action The animation to start
		 */
		public void startAction(String action) {
			animationState = getSpritesheet().startAction(action,  direction);
			millisecondsUntilChange = animationState.getDelay();
		}
		
		/**
		 * Computes the spritesheet for the layer
		 * 
		 * @return The layer's spritesheet
		 */
		public Spritesheet getSpritesheet() {
			return DyedSpritesheetRepository.get().getSpritesheet(name, dyeArray);
		}
	}
}
