package de.uni_frankfurt.prgpr.phase3.images;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Arrays;
import java.util.Map;

/**
 * Stores dyed spritesheets for later re-use, indexded by spritesheet name and dye specification.
 * 
 * This repository allows trading off in-memory representation size against run-time cost:
 * by incurring a small run-time cost to look up spritesheets here (rather than storing a reference
 * directly), data structures that reference spritesheets can over the network.
 * 
 * @author creichen
 *
 */
public class DyedSpritesheetRepository {
	Map<String, Map<List<DyeSpec>, Spritesheet>> spritesheetCache = new HashMap<>();

	private DyedSpritesheetRepository() {}
	
	private static DyedSpritesheetRepository SINGLETON = new DyedSpritesheetRepository();
	
	/**
	 * Retrieves the sole spritesheet repository
	 * 
	 * @return The SpritesheetRepository
	 */
	public static DyedSpritesheetRepository get() {
		return SINGLETON; 
	}
	
	/**
	 * Loads a spritesheet by name and dye specification.
	 * 
	 * Results are cached, meaning that calling this method with the same parameters multiple times is efficient
	 * (unlike repeated dyeing).
	 * 
	 * @param name Name of the spritesheet to load
	 * @param dyes Dyes to apply to the spritesheet
	 * @return The spritesheet, as indicated by its name and dyes
	 */
	public Spritesheet getSpritesheet(String name, DyeSpec ... dyes)  {
		List<DyeSpec> dyespec = Arrays.asList(dyes);
		Collections.sort(dyespec);
		if (!spritesheetCache.containsKey(name)) {
			spritesheetCache.put(name, new HashMap<List<DyeSpec>, Spritesheet>());
		}
		Spritesheet sheet = spritesheetCache.get(name).get(dyespec);
		if (sheet == null) {
			try {
				sheet = ImageLoader.defaultLoader().getSpritesheet(name).dye(dyes);
			} catch (IOException exn) {
				throw new RuntimeException(exn);
			}
			spritesheetCache.get(name).put(dyespec, sheet);
		}
		return sheet;
	}
}
