package de.uni_frankfurt.prgpr.phase3.images;

import java.io.Serializable;

/**
 * Represents the animation state in a given animation
 * 
 * @author creichen
 *
 */
public class AnimationState implements Serializable {
	private static final long serialVersionUID = 1L;

	/**
	 * 
	 */
	private final Animation animation;
	
	int sequenceIndex, sequencePos;
	boolean finished = false;

	/**
	 * Constructs a fresh animation state
	 * @param animation TODO
	 */
	public AnimationState(Animation animation) {
		this.animation = animation;
		reset();
	}
	
	/**
	 * Retrieves the current cel sequence
	 * 
	 * @return The current cel sequence
	 */
	private CelSequence seq() {
		return this.animation.getCelSequence(sequenceIndex);
	}
	
	/**
	 * Resets the animation state to the first index
	 */
	public void reset() {
		sequenceIndex = 0;
		sequencePos = seq().getStart();
	}
	
	/**
	 * Determines the horizontal (x) offset that the cel should be drawn to relative to the base coordinates
	 * 
	 * @return The horizontal offset
	 */
	public int getXOffset() {
		return seq().getXOffset();
	}

	/**
	 * Determines the vertical (y) offset that the cel should be drawn to relative to the base coordinates
	 * 
	 * @return The vertical offset
	 */
	public int getYOffset() {
		return seq().getYOffset();
	}

	/**
	 * Determines the amount of time that should elapse before the next animation step
	 * 
	 * @return Amount of time to elapse, in milliseconds
	 */
	public int getDelay() {
		return this.animation.getCelSequence(sequenceIndex).getDelay();
	}
	
	/**
	 * Gets the index of the cel that should be drawn for the current state of the animation
	 * 
	 * @return The cel index
	 */
	public int getCel() {
		if (isFinished()) {
			return -1;
		}
		return sequencePos;
	}
	
	/**
	 * Advances the animation by one step
	 * 
	 * @return false iff the animation is finished and there are no more steps
	 */
	public boolean next() {
		if (finished) {
			return false;
		}
		if (sequencePos == seq().getEnd()) {
			sequenceIndex++;
			if (sequenceIndex == this.animation.getCelSequencesNr()) {
				if (this.animation.isLooping()) {
					reset();
				} else {
					System.err.println("finished");
					finished = true;
					return false;
				}
			}
			sequencePos = seq().getStart();
		} else {
			sequencePos++;
		}
		return true;
	}
	
	/**
	 * Determines whether the animation is finished
	 * 
	 * @return true iff the animation has been completed
	 */
	public boolean isFinished() {
		return this.finished;
	}
}