package de.uni_frankfurt.prgpr.phase3.gui;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.concurrent.ConcurrentLinkedQueue;

import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import javax.swing.JTextField;
import javax.swing.text.BadLocationException;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledDocument;

/**
 * Main GUI interface.  Consists of three parts:
 * 
 * <ul>
 *  <li> canvas, for drawing graphics </li>
 *  <li> messageArea, for displaying messages </li>
 *  <li> inputField, for reading user text input</li>
 * </ul>
 * 
 * @author creichen
 */
public class UIPanel {
	// UIPanel sizes are fixed:
	public static final int TOTAL_WIDTH = 800;
	public static final int TOTAL_HEIGHT = 600;
	public static final int CANVAS_HEIGHT = 500;
	public static final int MAX_MESSAGE_AREA_TEXT = 65536; // maximum amount of text we retain in the message area
	public static final long TICKER_DELAY = 1L; // milliseconds between each tick event
	
	private UIEventDelegate eventObserver; // processes UI events
	private Drawer canvasDrawer; // redraws the canvas
	
	private ConcurrentLinkedQueue<AWTEvent> eventQueue = new ConcurrentLinkedQueue<>();
	
	private JFrame frame = null;
	private String windowTitle;
	private Canvas canvas;
	
	private JTextField inputField;
	private boolean inputFieldHasFocus = false; // does the input field have keyboard focus?  We use this to re-route events
	
	private JTextPane messageArea;
	private StyledDocument messages;
	
	private Thread tickerThread;
	
	// event management
	private CanvasEventListener asynchronousEventListener;
	private ActionListener synchronousInputListener;
	
	/**
	 * Creates a fresh InterfacePanel
	 * @param title Window title for the UI panel
	 */
	public UIPanel(String title) {
		this.windowTitle = title;
	}

	/**
	 * Sets the object that should process events observed by the UIPanel 
	 * 
	 * @param em The event manager that handles any events received by the panel
	 */
	public synchronized void setEventDelegate(UIEventDelegate em) {
		this.eventObserver = em;
	}
	
	/**
	 * Sets the object that will redraw the canvas whenever needed
	 * 
	 * @param d
	 */
	public synchronized void setDrawer(Drawer d) {
		this.canvasDrawer = d;
	}

	/**
	 * Adds message text to the message area.  Convenience method for black text.
	 * 
	 * See {@link #addColoredMessage(String, Color) addColoredMessage} for details. 
	 * 
	 * @param message The message to add.
	 */
	public void	addMessage(String message) {
		addColoredMessage(message, Color.BLACK);
	}
	    
	/**
	 * Adds message text to the message area.
	 * 
	 * To add a line break, use "\n".
	 * 
	 * This method will automatically remove excess text if the area is becoming too full.
	 * 
	 * @param message The message to add.
	 * @param color The colour to use for the message, e.g., Color.BLUE or Color.RED.
	 */
	public void	addColoredMessage(String message, Color color) {
		StyleContext styleContext = new StyleContext();
		Style style = styleContext.addStyle("", null);
		StyleConstants.setForeground(style, color);
	    try {
	    	messages.insertString(messages.getLength(), message, style);
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
	    if (messages.getLength() > MAX_MESSAGE_AREA_TEXT) {
	    	try {
				messages.remove(0,  messages.getLength() - MAX_MESSAGE_AREA_TEXT);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
	    }
    	messageArea.setCaretPosition(messages.getLength());
	}

	/**
	 * Focuses input on the input field.  This allows users to write text input.
	 */
	public void setInputFieldFocus() {
		this.inputField.requestFocus();
		this.inputFieldHasFocus = true;
	}
	
	/**
	 * Focuses input on the canvas.  This allows us to capture regular keyboard events.
	 */
	public void setCanvasFocus() {
		this.canvas.requestFocus();
		this.inputFieldHasFocus = false;
	}
	
	/**
	 * Sets up the JFrame used to display this interface and prepares it for
	 * interactive use
	 */
	//protected void setup() {
	public void setup() { // MUHAHAHAH EVIL GUY :)
		if (this.frame != null) {
			return;  // already set up
		}
		this.frame = new JFrame(this.windowTitle);
	    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
	    
	    Container pane = frame.getContentPane();
	    frame.setLayout(new BorderLayout());
	    
	    // Top: drawing canvas
	    this.canvas = new Canvas();
	    pane.add(canvas, BorderLayout.PAGE_START);
	    this.canvas.setPreferredSize(new Dimension(TOTAL_WIDTH, CANVAS_HEIGHT));
	    // enable event processing
	    this.asynchronousEventListener = new CanvasEventListener();
	    this.canvas.addKeyListener(asynchronousEventListener);
	    this.canvas.addMouseListener(asynchronousEventListener);
	    
	    // Bottom: input pane
	    this.inputField = new JTextField();
	    pane.add(inputField, BorderLayout.PAGE_END);
	    this.installAsynchronousStringInputEventListener(); // process string inputs from the input field
	    
	    // Middle: text messages
	    this.messageArea = new JTextPane();
	    messageArea.setEditable(false);
	    messages = messageArea.getStyledDocument();
	    JScrollPane messageAreaScrollPane = new JScrollPane(messageArea);
	
	    pane.add(messageAreaScrollPane, BorderLayout.CENTER); 
	    frame.setPreferredSize(new Dimension(TOTAL_WIDTH, TOTAL_HEIGHT));
	    frame.setExtendedState(JFrame.MAXIMIZED_BOTH);
	    frame.setResizable(false);
	    frame.pack(); // lay out
	}
	
	/**
	 * Reads the input of the input field into a StringInputEvent and clears the input field
	 * 
	 * @return A StringInputEvent capturing the input field's contents
	 */
	private StringInputEvent readOutInputField() {
		StringInputEvent retval = new StringInputEvent(inputField.getText());
		inputField.setText("");
		return retval;
	}
	
	/**
	 * For the input field: Installs the synchronous event handler, and uninstalls the asynchronous event handler.
	 * 
	 * This method prepares synchronous event processing for the input field.  We use it when
	 * manually dispatching events during queue processing.
	 */
	private synchronized void installSynchronousStringInputEventListener() {
		if (this.synchronousInputListener == null) {
			this.synchronousInputListener = new ActionListener() {

				@Override
				public void actionPerformed(ActionEvent e) {
					System.out.println("SYNCHRONOUS handler!");
					synchronousStringInputEvent = readOutInputField();
				}
			};
		}
		this.inputField.removeActionListener(this.asynchronousEventListener);
		this.inputField.addActionListener(this.synchronousInputListener);
	}

	/**
	 * For the input field: Installs the asynchronous event handler and uninstalls the synchronous event handler.
	 * 
	 * This method activates regular Swing-style event processing.  We only disable this mode while manually processing
	 *  events during `processEvents()'.
	 */
	private synchronized void installAsynchronousStringInputEventListener() {
		if (this.synchronousInputListener != null) {
			this.inputField.removeActionListener(this.synchronousInputListener);
		}
		this.inputField.addActionListener(this.asynchronousEventListener);
	}
	
	/**
	 * Synchronous string event.  This is used when manually dispatching events in processEvents.
	 */
	private StringInputEvent synchronousStringInputEvent = null;
	
	/**
	 * Retrieves and resets the synchronous string input event, if any.
	 * 
	 * The synchronus string input event is only set during synchronous string processing.  Otherwise, regular (asynchronous)
	 * Swing-style event processing is used.
	 * 
	 * @return null, or the synchronous string input event.
	 */
	private StringInputEvent getSyncronousStringInputEvent() {
		StringInputEvent retval = synchronousStringInputEvent;
		synchronousStringInputEvent = null;
		return retval;
	}
	
	/**
	 * Empties the event queue
	 */
	private synchronized void processEvents() {
		if (this.eventObserver == null) {
			// no event processing is taking place
			this.eventQueue.clear();
			return;
		}
		AWTEvent event;
		this.installSynchronousStringInputEventListener();
		
		// Iterate over all events in the queue.
		// Some of the events might be aimed at the input field.  If so, dispatch them to that field
		// and use the synchronus event listener to check for input field events-- those will then
		// interrupt our `regular' event flow.
		while ((event = eventQueue.poll()) != null) {
			while (event != null) {
				if (event instanceof MouseEvent) {
					MouseEvent mouseEvent = (MouseEvent) event;
					this.eventObserver.observeMouseClick(mouseEvent);
				} else if (event instanceof StringInputEvent) {
					eventObserver.observeTextInput(((StringInputEvent)event).getString());
				} else if (event instanceof KeyEvent) {
					KeyEvent keyEvent = (KeyEvent) event;
					switch (keyEvent.getID()) {
					case KeyEvent.KEY_TYPED:
						// due to our delayed event processing, events targetted at the input field may be delivered to the canvas instead
						if (inputFieldHasFocus) { // should've delivered this to the input field in the first place!
							inputField.dispatchEvent(keyEvent);
						}
						break;
					case KeyEvent.KEY_PRESSED:
						if (inputFieldHasFocus) {
							inputField.dispatchEvent(keyEvent);
							break;
						}
						this.eventObserver.observeKeyPress(keyEvent);
						if (inputFieldHasFocus) { // focus switch
							AWTEvent nextEvent = eventQueue.peek();
							if (nextEvent != null && nextEvent instanceof KeyEvent && nextEvent.getID() == KeyEvent.KEY_TYPED && ((KeyEvent) nextEvent).getKeyChar() == keyEvent.getKeyChar()) {
								eventQueue.poll(); // discard next event if we'd send an erroneous KEY_TYPED event to the inputField
							}
						}
						break;
					case KeyEvent.KEY_RELEASED:
						if (inputFieldHasFocus) {
							inputField.dispatchEvent(keyEvent);
							break;
						}
						this.eventObserver.observeKeyRelease(keyEvent);
						break;
					}
				}
				event = getSyncronousStringInputEvent(); // We might have finished a string input event in the mean time...
			}
		}
		this.installAsynchronousStringInputEventListener();
	}

	/**
	 * Starts the ticker thread, which updates the UIEventObserver every TICKER_DELAY milliseconds 
	 */
	private void startTickerThread() {
		if (tickerThread != null) {
			return;
		}
		tickerThread = new Thread() {
			@Override
			public void run() {
				long lastTime = System.currentTimeMillis();
				int tickCount = 0;
				
				while (true) {
					processEvents();
					repaint();
					synchronized(UIPanel.this) {
						if (eventObserver != null) {
							eventObserver.observeTick(tickCount);
						}
					}
					long currentTime = System.currentTimeMillis();
					long elapsed = currentTime - lastTime;
					long tickerIncrement = (elapsed + (TICKER_DELAY - 1)) / TICKER_DELAY;
					if (tickerIncrement == 0) {
						tickerIncrement = 1;
					}
					tickCount += tickerIncrement;
					long nextTime = lastTime + (tickerIncrement * TICKER_DELAY);
							
					currentTime = System.currentTimeMillis();
					while (currentTime < nextTime) {
						long sleepTime = nextTime - currentTime;
						try {
							Thread.sleep(sleepTime);
						} catch (Exception _) {
						}
						currentTime = System.currentTimeMillis();
					}
					lastTime = nextTime;
				}
			}
		};
		tickerThread.start();
	}
	
	/**
	 * Forces an image repaint
	 */
	public void repaint() {
		if (frame != null) {
			frame.repaint();
		}
	}
	
	/**
	 * Displays the UIPanel and starts event processing.
	 */
	public void show() {
		this.setup();
		this.startTickerThread();
	    frame.setVisible(true);
	}
	
	/**
	 * Canvas class, for drawing graphics
	 * 
	 * @author creichen
	 */
	private class Canvas extends JPanel {
		private static final long serialVersionUID = 1L;

		/**
		 * Creates a new canvas object
		 */
		public Canvas() {
			super(true);
			//super.setBackground(new Color(0, 128, 0)); // middle green
			// super.setBackground(new Color(198, 227, 240)); // middle green
			super.setBackground(new Color(0, 0, 0)); // middle black :)
		}
		
		/**
		 * Redraws the canvas
		 * 
		 * @param g The graphics context to redraw with
		 */
		@Override
		public void paintComponent(Graphics g) {
			super.paintComponent(g);
			
			synchronized (UIPanel.this) {
				if (canvasDrawer != null) {
					canvasDrawer.draw(g);
				}
			}

		}
	
	}
	
	/**
	 * Listens to canvas events and records them in the event queue
	 * 
	 * @author creichen
	 */
	private class CanvasEventListener implements KeyListener, MouseListener, ActionListener {

		/**
		 * Observe mouse click
		 * 
		 * @param arg0 Mouse button and coordinate details
		 */
		@Override
		public void mouseClicked(MouseEvent arg0) {
			synchronized (UIPanel.this) {
				eventQueue.add(arg0);
			}
		}

		@Override
		public void mouseEntered(MouseEvent arg0) {
		}

		@Override
		public void mouseExited(MouseEvent arg0) {
		}

		@Override
		public void mousePressed(MouseEvent arg0) {
		}

		@Override
		public void mouseReleased(MouseEvent arg0) {
		}

		/**
		 * Observe key press
		 * 
		 * @param arg0 key details
		 */
		@Override
		public void keyPressed(KeyEvent arg0) {
			synchronized (UIPanel.this) {
				eventQueue.add(arg0);
			}
		}

		/**
		 * Observe key release
		 * 
		 * @param arg0 key details
		 */
		@Override
		public void keyReleased(KeyEvent arg0) {
			synchronized (UIPanel.this) {
				eventQueue.add(arg0);
			}
		}

		/**
		 * Observe key being typed; this is subsumed by keyPressed and keyReleased.
		 * 
		 * We separately record these events in order to report them to the inputPanel, if needed.
		 * 
		 * @param arg0 key details
		 */
		@Override
		public void keyTyped(KeyEvent arg0) {
			synchronized (UIPanel.this) {
				eventQueue.add(arg0);
			}
		}

		/**
		 * Observe input field action
		 * 
		 * @param action The action (ignored)
		 */
		@Override
		public void actionPerformed(ActionEvent action) {
			synchronized (UIPanel.this) {
				eventQueue.add(readOutInputField());
			}
		}
	}

	/**
	 * Event that captures string input via the input field
	 * @author creichen
	 *
	 */
	private class StringInputEvent extends AWTEvent {
		private static final long serialVersionUID = 1L;
		private String body;

		/**
		 * Constructs a new string input event
		 * 
		 * @param s String to contain in the event
		 */
		public StringInputEvent(String s) {
			super("", 0);
			this.body = s;
		}
		
		/**
		 * Retrieves the event payload
		 * 
		 * @return The string contained in the event
		 */
		public String getString() {
			return this.body;
		}
	}
}
