package de.uni_frankfurt.prgpr.core;

import java.io.Serializable;

/**
 * Message to or from the server.
 * 
 * @author creichen
 *
 */
public abstract class Message implements Serializable {

	private static final long serialVersionUID = 3402940003310824248L;
	
	/**
	 * Name of the project this message is intended for
	 */
	private String project;
	
	public Message()
	{
	}
	
	/**
	 * Checks that all required fields are set.  Otherwise throws a RuntimeException.
	 */
	public void validate()
	{
		if (project == null) {
			throw new RuntimeException("Must call setProject(...) with non-null parameter before sending message!");
		}
	}

	/**
	 * @return The project that this message is intended for
	 */
	public String getProject() {
		return project;
	}

	/**
	 * Sets the project name
	 * 
	 * @param project The project that this message is intended for
	 */
	public void setProject(String project) {
		this.project = project;
	}

	/**
	 * Authenticated server access request, including project name and project login.
	 *  
	 * @author creichen
	 *
	 */
	public abstract static class Login extends Message {
		private static final long serialVersionUID = 1067723956075195665L;
		
		private String password;

		/**
		 * @return the stored password
		 */
		public String getPassword() {
			return password;
		}

		/**
		 * Sets the password
		 * @param password The password
		 */
		public void setPassword(String password) {
			this.password = password;
		}

		public void validate()
		{
			super.validate();
//			if (password != null) {
//			  	throw new RuntimeException("Must call setPassword(...) with non-null parameter before sending this kind of message!");
//			}
		}
	}
	
	/**
	 * Code transfer message.
	 * 
	 * @author creichen
	 *
	 */
	public abstract static class Code extends Login {
		private static final long serialVersionUID = 2892649484372587614L;
		
		private byte[] data;
		private String clientMainClass;

		/**
		 * Gets the contents of the jar file representing the project's Java bytecode
		 * @return The byte contents
		 */
		public byte[] getJarData() {
			return data;
		}

		/**
		 * Sets the contents of the jar file containing all relevant code 
		 * @param data The bytes in the jar file
		 */
		public void setJarData(byte[] data) {
			this.data = data;
		}

		/**
		 * Sets the main class for the client to use in this project
		 * 
		 * @return The main class for the client
		 */
		public String getClientMainClass() {
			return clientMainClass;
		}

		/**
		 * Retrieves the client's main class to invoke for the project.
		 * @param mainClass Name of the main class
		 */
		public void setClientMainClass(String mainClass) {
			this.clientMainClass = mainClass;
		}
		
		public void validate()
		{
			super.validate();
			if (data == null) {
				throw new RuntimeException("Must call setJarData(...) with non-null parameter before sending this kind of message!");
			}
			if (clientMainClass == null) {
				throw new RuntimeException("Must call setMainClass(...) with non-null parameter before sending this kind of message!");
			}
		}
	}
	
	/**
	 * Request to server to retrieve client data for given project.
	 * @author creichen
	 *
	 */
	public static class GetClient extends Message {
		private static final long serialVersionUID = 8745439188087549985L;
		
		@Override
		public String toString()
		{
			return "GET-CLIENT(" + getProject() + ")";
		}
	}
	
	/**
	 * Response from server to send client code for given project.  Password field is kept empty. 
	 * @author creichen
	 *
	 */
	public static class Client extends Code {
		public Client() {
			setPassword("");
		}
		
		private static final long serialVersionUID = 8745439188087549985L;
		
		@Override
		public String toString()
		{
			return "CLIENT(" + getProject() + ", data*" + getJarData().length + ", cl-main=" + getClientMainClass() + ")"; 
		}
	}
		
	/**
	 * Request to server to update given project.  This includes both client and server data.
	 * @author creichen
	 *
	 */
	public static class Update extends Code {
		private static final long serialVersionUID = 8745439188022549985L;
		
		private String serverMainClass;
		private int version = -1;
		
		/**
		 * Gets the name of the main server class to invoke.
		 * @return Name of the main class
		 */
		public String getServerMainClass() {
			return serverMainClass;
		}
		
		/**
		 * Sets the name of the main server class.
		 * @param mainClass
		 */
		public void setServerMainClass(String mainClass) {
			this.serverMainClass = mainClass;
		}
		
		/**
		 * Gets the version of the update.
		 * @return Update version.
		 */
		public int getVersion() {
			return version;
		}
		
		/**
		 * Sets the version of the project.  The version must increase on every update, unless it is EXPERIMENTAL.
		 * @param version
		 */
		public void setVersion(int version) {
			this.version = version;
		}
		
		public void validate()
		{
			super.validate();
			if (serverMainClass == null) {
				throw new RuntimeException("Must call setServerMainClass(...) with non-null parameter before sending this kind of message!");
			}
			if (version < 0) {
				throw new RuntimeException("Must call setVersion(...) with non-negative parameter before sending this kind of message!");
			}
		}
		
		@Override
		public String toString()
		{
			return "UPDATE(" + getProject() + ", v" + this.version + ", data*" + getJarData().length + ", srv-main=" + getServerMainClass() + ", cl-main=" + getClientMainClass() + ")"; 
		}
	}
	
	/**
	 * Request establishing a connection to server
	 * @author creichen
	 *
	 */
	public static class Connect extends Message {
		private static final long serialVersionUID = 398982313198501708L;
		
		/**
		 * Constructs a new connection request 
		 */
		public Connect() {
		}
		
		@Override
		public String toString() {
			return "CONNECT";
		}
	}
	
	/**
	 * Error response sent by server. 
	 * @author creichen
	 *
	 */
	public static class Error extends Message {
		private static final long serialVersionUID = 3989854283178501708L;
		
		private String message;

		/**
		 * Creates an error message
		 * @param message
		 */
		public Error(String message) {
			this.message = message;
		}
		
		/**
		 * Obtains the error message
		 * @return Error message
		 */
		public String getError() {
			return this.message;
		}
		
		public String toString() {
			return "Error: " + this.getError();
		}
	}
	
	/**
	 * Confirmation response sent by server.
	 * @author creichen
	 *
	 */
	public static class OK extends Message {
		private static final long serialVersionUID = 3989851233178501708L;
		
		public OK()
		{
		}

		public String toString() {
			return "OK";
		}
	}
}
