package Pacman;

/**
 * @author Daniel
 *         the enemy class represents the ghosts in our pacman game
 *         the AI has two modes, the random-way-to-go and the looking-way-to-go
 *         (hard-mode)
 *         enemies do chase players in order to catch them
 *         enemies can be eaten if the chaos-mode of the map they're spawned on
 *         is enabled
 *         enemies respawn after time at a fix point
 *         enemies don't take pills
 */
public class Enemy implements Tile {
	/** the direction the enemy is facing towards **/
	private String dir = "left";
	/** the name of the enemy (blinky, pinky, inky, clyde) **/
	public String name;
	/** the current amount of accessable tiles around the enemy on the map **/
	private int index = 0;
	/** the x position of the enemy on the map **/
	public short x;
	/** the x position of the enemy on the map **/
	public short y;
	/**
	 * true, if the enemy is alive and may move, false if it has been caught by
	 * a player
	 **/
	public boolean alive;
	/** the time it takes until the enemy will be respawned **/
	public int deathLasting = 0;
	/** is true, if the enemy stepped on a pill, false if not **/
	private boolean onPill;
	/** is true, if the enemy stepped on a power-pill, false if not **/
	private boolean onPower;
	/** an array to save the available directions to go **/
	private String[] dirs = new String[3];
	/** the intellgence of the enemy **/
	private String status;
	/** is true if the enemy is looking and has spotted an player nearby **/
	private boolean hasTarget = false;
	/** the map it is spawned on **/
	private Map map;

	/**
	 * constructor or the enemy-class
	 * 
	 * @param name
	 *            the name the enemy shoudl have (should not be anything else
	 *            than {"blinky", "pinky", "inky", "clyde"})
	 * @param intelligence
	 *            the intellgence of the enemy (should not be anything else than
	 *            {"random", "looking"})
	 */
	public Enemy(String name, String intelligence) {
		this.name = name;
		this.status = intelligence;
	}

	/**
	 * makes the enemy move on the map
	 * called regularly by the main method, or by the server's update
	 * 
	 */
	public void move() {
		try {
			if (map.chaos) { //in case of map.chaos == true, we want to make the enemy move randomly. it will end up as an "easy-snack", which is equivalent to slowing it down
				String[] randomDir = { "up", "right", "left", "down" };
				dir = randomDir[(int) (Math.random() * randomDir.length)]; //just choose a random direction to go
				for (int i = 0; i < 3; i++) {
					if (ahead(dir))
						continue; //take care of not running into a wall
					dir = randomDir[(int) (Math.random() * randomDir.length)];
				}
			}
			if (status.equals("random")) {
				setDirs();
				dir = dirs[(int) (Math.random() * index)]; //just pick a random available direction to go, without turning around
				deadEnd();
				move(dir);
				index = 0;
			} else if (status.equals("looking")) {
				setDirs();
				if (index > 1) //if the enemy passes a crossing, it checks of a player is in sight
					checkForPlayer();
				else
					turn();
				deadEnd();
				move(dir);
				index = 0;
			}
		} catch (Exception e) {
			dir = "up"; //we might run into a nullpointer for some reason :) so just catch it by setting the direction to go anywhere but null
		}
	}

	/**
	 * checks all tiles around the enemy, if one is accessable, just add it to
	 * the array which stores possible directions to go
	 */
	private void setDirs() {
		if (ahead("left") && !(dir.equals("right"))) {
			dirs[index] = "left";
			++index;
		}
		if (ahead("right") && !(dir.equals("left"))) {
			dirs[index] = "right";
			++index;
		}
		if (ahead("up") && !(dir.equals("down"))) {
			dirs[index] = "up";
			++index;
		}
		if (ahead("down") && !(dir.equals("up"))) {
			dirs[index] = "down";
			++index;
		}
	}

	/**
	 * checks if the enemy runs into a dead-end it will then just turn around
	 */
	private void deadEnd() {
		if (!ahead(dir)) {
			if (dir.equals("up")) {
				if (!ahead("left") && !(ahead("right"))) {
					dir = "down";
				}
			} else if (dir.equals("down")) {
				if (!ahead("left") && !(ahead("right"))) {
					dir = "up";
				}
			} else if (dir.equals("left")) {
				if (!ahead("up") && !(ahead("down"))) {
					dir = "right";
				}
			} else if (dir.equals("right")) {
				if (!ahead("up") && !(ahead("down"))) {
					dir = "left";
				}
			}
		}
	}

	/**
	 * 
	 * moves the enenmy to a given direcion
	 * 
	 * @param dir
	 *            the direction for the enemy to go to
	 */
	public void move(String dir) {
		switch (dir) {
		case "up": //this will translate the direction into x and y position changes.
			move(0, -1); //for "up", we need to decrement the x-pos, and leave the y-pos unchanged
			break;
		case "down":
			move(0, 1); //and so on...
			break;
		case "right":
			move(1, 0);
			break;
		case "left":
			move(-1, 0);
			break;
		}
	}

	/**
	 * checks of the tile in front of the enemy is accessable or not
	 * 
	 * @param dir
	 *            the direction the enemy is facing
	 * @return true, if the tile the enemy is facing is accessable, false if not
	 */
	private boolean ahead(String dir) {
		switch (dir) {
		case "up": //Walls and other Enemies are the only tiles not accessable. Players, Air, Bonus and Pills are accessable 
			return !(map.getCel(this.x, this.y - 1) instanceof Wall || map.getCel(this.x, this.y - 1) instanceof Enemy);
		case "down":
			return !(map.getCel(this.x, this.y + 1) instanceof Wall || map.getCel(this.x, this.y + 1) instanceof Enemy);
		case "right":
			return !(map.getCel(this.x + 1, this.y) instanceof Wall || map.getCel(this.x + 1, this.y) instanceof Enemy);
		case "left":
			return !(map.getCel(this.x - 1, this.y) instanceof Wall || map.getCel(this.x - 1, this.y) instanceof Enemy);
		}
		return true;
	}

	/**
	 * sets the intelligence of an enemy
	 * 
	 * @param status
	 *            the intelligence the enemy shall have
	 */
	public void setIntelligence(String status) {
		this.status = status;
	}

	/**
	 * checks if there are any players visible at this crossing
	 */
	private void checkForPlayer() {
		hasTarget = false; //checks if there are closer players around visible
		for (int i = 0; i < index; i++) { //check all available directions at this crossing
			int dist; //the dist it take to find a player or run into a wall
			switch (dirs[i]) { //is up an available direction to look for players?
			case "up":
				dist = 1; //start by looking at the first tile in "up"-direction
				while (!(map.getCel(this.x, this.y - dist) instanceof Wall)) { //keep searching upwards if we hadn't run into a player or wall yet
					if (map.getCel(this.x, this.y - dist) instanceof Player) { //did we find a player?
						dir = "up";
						hasTarget = true;
					}
					dist++;
				}
			case "down": //same goes for the other directions...
				dist = 1;
				while (!(map.getCel(this.x, this.y + dist) instanceof Wall)) {
					if (map.getCel(this.x, this.y + dist) instanceof Player) {
						dir = "down";
						hasTarget = true;
					}
					dist++;
				}
			case "left":
				dist = 1;
				while (!(map.getCel(this.x - dist, this.y) instanceof Wall)) {
					if (map.getCel(this.x - dist, this.y) instanceof Player) {
						dir = "left";
						hasTarget = true;
					}
					dist++;
				}
			case "right":
				dist = 1;
				while (!(map.getCel(this.x + dist, this.y) instanceof Wall)) {
					if (map.getCel(this.x + dist, this.y) instanceof Player) {
						dir = "right";
						hasTarget = true;
					}
					dist++;
				}
			}
		}
		if (!hasTarget) //we weren't successful at this crossing... just move randomly
			dir = dirs[(int) (Math.random() * index)];

	}

	/**
	 * handles turning if the enemy runs into a 90 dergree curve
	 */
	private void turn() {
		if (!ahead(dir)) {
			dir = dirs[0];
		}
	}

	/**
	 * 
	 * the main method to make the enemy move
	 * it handles stepping on pill, players, walls, power, bonus, etc...
	 * 
	 * @param x
	 *            is -1 for left, 1 for right
	 * @param y
	 *            is -1 for up, 1 for down
	 */
	public void move(int x, int y) {

		try {
			if (ahead(dir)) { //are we facing a wall? this should no longer occur since setDirs works correctly
				if (onPill) { //are we currently upon a pill - it should not be removed
					if (!onPower) { //is this pill we are standing on a power pill?
						if (map.getCel(this.x + x, this.y + y) instanceof Pill) { //handle movement upon a pill
							onPower = (((Pill) map.getCel(this.x + x, this.y + y)).power); //did we step upon a power pill?
							map.setCel(this.x + x, this.y + y, this); //set the facing tile to a  instance of this object (this will actually make this object move forward)
							map.setCel(this.x, this.y, new Pill(false)); //set the tile we left behind
							this.x += x; //adjust x, y positions of this object
							this.y += y;
						} else if (map.getCel(this.x + x, this.y + y) instanceof Air //keep going in that way for all the other instances of tiles we might step on
								|| map.getCel(this.x + x, this.y + y) instanceof Bonus) {
							map.setCel(this.x + x, this.y + y, this);
							map.setCel(this.x, this.y, new Pill(false));
							onPill = false;
							this.x += x;
							this.y += y;
						} else if (map.getCel(this.x + x, this.y + y) instanceof Player && !map.chaos) { //enemies can't eat a player if map.chaos is true
							onPill = false;
							if (((Player) map.getCel(this.x + x, this.y + y)).alive) //can't kill ghost-players, that are already dead and therefore can't move
								map.players--; //make the map know that a player is no longer alive
							((Player) map.getCel(this.x + x, this.y + y)).alive = false;
							map.setCel(this.x + x, this.y + y, this);
							map.setCel(this.x, this.y, new Pill(false));
							this.x += x;
							this.y += y;
						}
					} else { //same handling if we are on a power-pill
						if (map.getCel(this.x + x, this.y + y) instanceof Pill) {
							map.setCel(this.x, this.y, new Pill(onPower));
							onPower = (((Pill) map.getCel(this.x + x, this.y + y)).power);
							map.setCel(this.x + x, this.y + y, this);
							this.x += x;
							this.y += y;
						} else if (map.getCel(this.x + x, this.y + y) instanceof Air
								|| map.getCel(this.x + x, this.y + y) instanceof Bonus) {
							map.setCel(this.x + x, this.y + y, this);
							map.setCel(this.x, this.y, new Pill(onPower));
							onPower = false;
							onPill = false;
							this.x += x;
							this.y += y;
						} else if (map.getCel(this.x + x, this.y + y) instanceof Player && !map.chaos) {
							onPill = false;
							if (((Player) map.getCel(this.x + x, this.y + y)).alive)
								map.players--;
							((Player) map.getCel(this.x + x, this.y + y)).alive = false;
							map.setCel(this.x + x, this.y + y, this);
							map.setCel(this.x, this.y, new Pill(true));
							this.x += x;
							this.y += y;
						}
					}
				} else { //the enemy did not step upon a pill - same handling as above, but this time we don't need to take abount placing the pill we stood on
					if (map.getCel(this.x + x, this.y + y) instanceof Pill) {
						onPower = (((Pill) map.getCel(this.x + x, this.y + y)).power);
						map.setCel(this.x + x, this.y + y, this);
						map.setCel(this.x, this.y, new Air());
						onPill = true;
						this.x += x;
						this.y += y;
					} else if (map.getCel(this.x + x, this.y + y) instanceof Air) {
						map.setCel(this.x + x, this.y + y, this);
						map.setCel(this.x, this.y, new Air());
						onPill = false;
						this.x += x;
						this.y += y;
					} else if (map.getCel(this.x + x, this.y + y) instanceof Player && !map.chaos) {
						onPill = false;
						if (((Player) map.getCel(this.x + x, this.y + y)).alive)
							map.players--;
						((Player) map.getCel(this.x + x, this.y + y)).alive = false;
						map.setCel(this.x + x, this.y + y, this);
						map.setCel(this.x, this.y, new Air());
						this.x += x;
						this.y += y;
					} else if (map.getCel(this.x + x, this.y + y) instanceof Bonus) {
						map.setCel(this.x + x, this.y + y, this);
						map.setCel(this.x, this.y, new Air());
						map.randomBonusNr = (int) (Math.random() * 4);
						map.bonusCooldown = (map.bonusCooldown == 0) ? 60 : map.bonusCooldown;
						dirs[0] = "left"; //enemies should not be able to walk downwards when they step on the bonus tile
						dirs[1] = "right"; //the bonus tile is always located right above the exit of the enemy spawn
						dirs[2] = null;
						onPill = false;
						this.x += x;
						this.y += y;
					}
				}
			}

		} catch (Exception e) { //sometimes there might be an exception thrown for some reasons
		} //i don't know if they still appear, but for safety reasons just catch the exception and keep
	} //it unhandled in order to not let the game crash

	/**
	 * 
	 * sets the dir of the enemy. should be called by the client in multiplayer
	 * mode in order to draw the correct image on our canvas
	 * 
	 * @param dir
	 *            the direction the enemy should be looking
	 */
	public void setDir(String dir) {
		this.dir = dir;
	}

	/**
	 * @return the direction the enemy is looking currently
	 */
	public String getDir() {
		return dir;
	}

	/**
	 * makes the enemy die
	 * called if a player ate an enemy
	 */
	public void die() {
		this.alive = false;
		if (onPill) //make sure that the player which ate the enemy also collects the pill that was under the eaten enemy
			map.pills--;
		onPill = false;
		deathLasting = 25; //the amount of game-logic calls it takes until the enemy respawns
	}

	/**
	 * 
	 * spawns the enemy on an given map
	 * 
	 * @param x
	 *            the x-position where the enemy will spawn
	 * @param y
	 *            the y-position where the enemy will spawn
	 * @param map
	 *            the map on which the enemy will spawn
	 */
	public void spawn(int x, int y, Map map) {
		this.map = map;
		this.x = (short) x;
		this.y = (short) y;
		this.alive = true; //make it live
		map.setCel(x, y, this); //set the map-cel to this object, so we can see it
	}

}