package de.uni_frankfurt.prgpr.phase2.sprites;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;

import javax.imageio.ImageIO;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Class for loading spritesheets from a zip file
 * 
 * @author creichen
 *
 */
public class SpritesheetLoader {
	
	private String filename;

	private Map<String, ZipEntry> spritesheets = null;

	// XML parser
	private DocumentBuilder xmlParser = null;

	private ZipFile zipFile;
	
	/**
	 * Creates a spritesheet loader for the specified file name
	 * 
	 * @param filename Name of the zip file to load
	 */
	public SpritesheetLoader(String filename) {
		this.filename = filename;
		try {
			this.xmlParser = DocumentBuilderFactory.newInstance().newDocumentBuilder();
		} catch (ParserConfigurationException exn) {
			throw new RuntimeException(exn);
		}
	}

	/**
	 * Debug method to print XML documents
	 * 
	 * @param doc The XML document to print
	 * @param out The output stream to print to
	 */
	@SuppressWarnings("unused")
	private static void printDocument(Document doc, OutputStream out) {
		// code by http://stackoverflow.com/users/203907/bozho
		try {
			TransformerFactory tf = TransformerFactory.newInstance();
			Transformer transformer = tf.newTransformer();
			transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "no");
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
			transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");

			transformer.transform(new DOMSource(doc), 
					new StreamResult(new OutputStreamWriter(out, "UTF-8")));
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Initialises the SpritesheetLoader by indexing all viable spritesheets and preparing them for loading
	 * @throws IOException if the file could not be loaded
	 */
	public void init() throws IOException {
		if (spritesheets != null) {
			return;
		}
		
		spritesheets = new HashMap<>();
		
		this.zipFile = new ZipFile(this.filename);
		Enumeration<? extends ZipEntry> zipIterator = zipFile.entries();
		while (zipIterator.hasMoreElements()) {
			ZipEntry z = zipIterator.nextElement();
			if (z.getName().endsWith(".xml")) {
				String name = z.getName().substring(0, z.getName().length() - 4);
				spritesheets.put(name, z);
			}
		}
	}
	
	/**
	 * Retrieves a spritesheet by name
	 * 
	 * @param name Name of the spritesheet
	 * @return A spritesheet, obtained from the given resource, or null on error
	 */
	public Spritesheet getSpritesheet(String name) {
		ZipEntry entry = spritesheets.get(name);
		if (entry == null) {
			return null;
		}
		SpritesheetParser parser = new SpritesheetParser();
		parser.parseToplevel(getData(entry));
		return parser.getSpritesheet();
	}
	
	/**
	 * Helper function to extract the bytes for a given ZipEntry
	 * 
	 * @param entry ZipEntry to extract the data for
	 * @return The input stream containing the given zip entry contents
	 */
	public InputStream getData(ZipEntry entry) {
		try {
			InputStream s = zipFile.getInputStream(entry);
			return s;
			
		} catch (IOException exn) {
			throw new RuntimeException(exn);
		}
	}
	
	/**
	 * Helper function to extract the bytes for a given ZipEntry
	 * 
	 * @param entry ZipEntry to extract the data for
	 * @return The input stream containing the given zip entry contents
	 */
	private InputStream getDataForFilename(String data) {
		try {
			ZipEntry entry = zipFile.getEntry(data);
			if (entry == null) {
				throw new RuntimeException("Missing file entry: " + data);
			}
			InputStream s = zipFile.getInputStream(entry);
			return s;
			
		} catch (IOException exn) {
			throw new RuntimeException(exn);
		}
	}

	/**
	 * Retrieves the set of all spritesheet names
	 * 
	 * @return A set of all spritesheet names
	 */
	public Set<? extends String> getSpritesheetNames() {
		return spritesheets.keySet();
	}
	
	private static SpritesheetLoader DEFAULT = null;
	
	/**
	 * Loads sprites from the default location
	 * @return A spritesheet loader from the default location
	 * @throws IOException
	 */
	public static SpritesheetLoader defaultLoader() throws IOException {
		if (DEFAULT != null) {
			return DEFAULT;
		}
		DEFAULT = new SpritesheetLoader(System.getProperty("user.home") + File.separator + "prgpr" + File.separator + "graphics.zip");
		System.out.println(System.getProperty("user.home") + File.separator + "prgpr" + File.separator + "graphics.zip");
		DEFAULT.init();
		return DEFAULT;
	}
	
	/**
	 * Class for parsing a single spritesheet
	 * @author creichen
	 *
	 */
	private class SpritesheetParser {
		private Spritesheet spritesheet;

		/**
		 * Retrieves the spritesheet built up by the spritesheet parser
		 * 
		 * @return The generated spritesheet
		 */
		public Spritesheet getSpritesheet() {
			return spritesheet;
		}
		
		/**
		 * Parses an XML input stream into an XML document element
		 * 
		 * @param indoc
		 * @return
		 */
		public Element parseDoc(InputStream indoc) {
			Document doc;
			try {
				doc = xmlParser.parse(indoc);
			} catch (SAXException e) {
				e.printStackTrace();
				return null;
			} catch (IOException e) {
				e.printStackTrace();
				return null;
			}
			return doc.getDocumentElement();
		}
		
		/**
		 * Checks if the element is an '&lt;include file="..."&gt;' element; if so, includes the specified file 
		 * @param elt The element to check
		 * @return The document element of the included specification, or null
		 */
		public Element includeIfNeeded(Element elt) {
			if (elt.getTagName().equals("include")) {
				return parseDoc(getDataForFilename(elt.getAttribute("file")));
			}
			return null;
		}
		
		/**
		 * Parses an &lt;imageset&gt; element
		 * 
		 * @param elt The element to parse
		 */
		private void parseImageSet(Element elt) {
			if (spritesheet != null) { // already have a spritesheet; obey override semantics
				return;
			}
			String filename = elt.getAttribute("src");
			EnumSet<DyeChannel> dyeChannels = EnumSet.noneOf(DyeChannel.class);
			int splitIndex;
			if ((splitIndex = filename.indexOf('|')) > 0) {
				int index = splitIndex + 1;
				for (; index < filename.length(); index++) {
					switch (filename.charAt(index)) {
					case 'R': dyeChannels.add(DyeChannel.RED); break;
					case 'G': dyeChannels.add(DyeChannel.GREEN); break;
					case 'B': dyeChannels.add(DyeChannel.BLUE); break;
					case 'C': dyeChannels.add(DyeChannel.CYAN); break;
					case 'M': dyeChannels.add(DyeChannel.MAGENTA); break;
					case 'Y': dyeChannels.add(DyeChannel.YELLOW); break;
					case 'W': dyeChannels.add(DyeChannel.WHITE); break;
					}
				}
				filename = filename.substring(0, splitIndex); 
			}
			BufferedImage bufferedImage;
			try {
				bufferedImage = ImageIO.read(getDataForFilename(filename));
			} catch (IOException e) {
				e.printStackTrace();
				throw new RuntimeException(e);
			}
			spritesheet = new Spritesheet(bufferedImage, dyeChannels);
			if (elt.hasAttribute("width") && elt.hasAttribute("height")) {
				spritesheet.setCelDimensions(Integer.parseInt(elt.getAttribute("width")), Integer.parseInt(elt.getAttribute("height")));
			}
		}
		
		int intAttribute(Element elt, String name) {
			if (elt.hasAttribute(name)) {
				return Integer.parseInt(elt.getAttribute(name));
			}
			return 0;
		}
		
		/**
		 * Parses an &lt;animation&gt; tag, which usually represents one direction of animation within a specific animation name
		 * 
		 * @param elt The animation to parse
		 * @param name Name of the animation
		 */
		private void parseAnimation(Element elt, String name) {
			Direction direction = Direction.DEFAULT;
			switch (elt.getAttribute("direction")) {
			case "up": direction = Direction.UP; break;
			case "down": direction = Direction.DOWN; break;
			case "left": direction = Direction.LEFT; break;
			case "right": direction = Direction.RIGHT; break;
			}
			
			Animation animation = new Animation();
			boolean looping = true;
			
			NodeList nl = elt.getChildNodes();
			
			for (int i = 0; i < nl.getLength(); i++) {
				Node n = nl.item(i);
				if (n instanceof Element) {
					Element animationStep = (Element)n;
					
					int xoffset = intAttribute(animationStep, "offsetX");
					int yoffset = intAttribute(animationStep, "offsetY");
					
					switch (animationStep.getTagName()) {
					case "frame":
						int index = intAttribute(animationStep, "index");
						animation.addCelSequence(new CelSequence(
								index, index, intAttribute(animationStep, "delay")).setOffsets(xoffset,  yoffset));
						break;
					case "sequence":
						animation.addCelSequence(new CelSequence(
								intAttribute(animationStep, "start"),
								intAttribute(animationStep, "end"),
								intAttribute(animationStep, "delay")).setOffsets(xoffset, yoffset));
						break;
					case "end":
						looping = false;
						break;
					}
				}
			}
			
			animation.setIsLooping(looping);
			spritesheet.addAnimation(name, direction, animation);
		}
		
		/**
		 * Parses an &lt;action&gt; tag
		 * 
		 * @param elt The element to parse
		 */
		private void parseAction(Element elt) {
			String actionName = elt.getAttribute("name");
			
			NodeList nl = elt.getChildNodes();
			
			for (int i = 0; i < nl.getLength(); i++) {
				Node n = nl.item(i);
				if (n instanceof Element) {
					Element actionPart = (Element)n;
					switch (actionPart.getTagName()) {
					case "animation": parseAnimation(actionPart, actionName); break;
					}
				}
			}
		}
			
		/**
		 * Parses a toplevel XML document input stream for a spritesheet specification
		 * 
		 * @param docStream The stream to parse from
		 */
		public void parseToplevel(InputStream docStream) {
			parseToplevelElement(parseDoc(docStream));
		}

		/**
		 * Parses a toplevel spritesheet definition XML document
		 * 
		 * @param body The document element for the spritesheet
		 */
		public void parseToplevelElement(Element body) {
			if (body == null) {
				return;
			}
			
			if (!body.getTagName().equals("sprite")) {
				System.err.println("Not a <sprite>");
				return;
			}
			NodeList nl = body.getChildNodes();
			for (int i = 0; i < nl.getLength(); i++) { 
				Node n = nl.item(i);
				if (n instanceof Element) {
					Element elt = (Element) n;
					switch (elt.getTagName()) {
					case "imageset":
						parseImageSet(elt);
						break;
					case "action":
						parseAction(elt);
						break;
					case "include":
						parseToplevelElement(includeIfNeeded(elt));
					}
				}
			}
		}
	}
}
