package de.uni_frankfurt.prgpr.phase2.sprites;

import java.awt.Color;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import de.uni_frankfurt.prgpr.phase2.gui.Gradient;

/**
 * Spritesheets represent a set of `cels', or animation frames, together with
 * `actions' that describe animations constructed from the cels.
 * 
 * Spritesheets may also be dyeable, meaning that parts of the spritesheet may
 * have particular colours swapped out by a colour gradient. This allows e.g. a
 * white `shirt' spritesheet to be re-used as a red, green, blue, or purple
 * spritesheet.
 * 
 * @author creichen
 *
 */
public class Spritesheet {
	private EnumSet<DyeChannel> dyeChannels;
	private BufferedImage image;
	private int celWidth, celHeight; // size of each individual cel
	private int celColumns; // number of total columns of cels in the
							// spritesheet
	private int celRows; // number of total rows of cels in the spritesheet
	private Map<String, Map<Direction, Animation>> animations = new HashMap<>(); // all
																					// animations

	/**
	 * Constructs a new spritesheet
	 * 
	 * @param img
	 *            The image that this spritesheet is based on
	 * @param dyes
	 *            The number of dye channels supported by the spritesheet, for
	 *            dyeing parts of the image
	 */
	public Spritesheet(BufferedImage img, EnumSet<DyeChannel> dyes) {
		this.image = img;
		this.dyeChannels = dyes;
		this.setCelDimensions(img.getWidth(), img.getHeight());
	}

	/**
	 * Checks whether the spritesheet can be dyed
	 * 
	 * @return true iff the spritesheet can be dyed
	 */
	public boolean isDyeable() {
		return this.dyeChannels.isEmpty();
	}

	/**
	 * Returns the set of all dyes that we can apply to this spritesheet
	 * 
	 * @return A set of all DyeChannels that we can set for this spritesheet
	 */
	public Set<? extends DyeChannel> getDyes() {
		return this.dyeChannels;
	}

	/**
	 * Sets the sizes of the individual cels stored in the spritesheet
	 * 
	 * Used during initialisation
	 * 
	 * @param width
	 *            The width of each cel
	 * @param height
	 *            The height of each cel
	 */
	void setCelDimensions(int width, int height) {
		this.celWidth = width;
		this.celHeight = height;
		this.celColumns = this.image.getWidth() / width;
		this.celRows = this.image.getHeight() / height;
	}

	/**
	 * Adds a single animation to the spritesheet
	 * 
	 * @param name
	 *            Name of the animation to add
	 * @param direction
	 *            Direction within that animation
	 * @param animation
	 *            The animation object describing the animation
	 */
	void addAnimation(String name, Direction direction, Animation animation) {
		Map<Direction, Animation> animationRecord;
		if (!this.animations.containsKey(name)) {
			animationRecord = new HashMap<>();
			this.animations.put(name, animationRecord);
		} else {
			animationRecord = this.animations.get(name);
		}
		animationRecord.put(direction, animation);
	}

	/**
	 * Lists all actions supported by this spritesheet
	 * 
	 * @return The set of all actions that this spritesheet can animate
	 */
	public Set<? extends String> getActions() {
		return this.animations.keySet();
	}

	/**
	 * Lists all directions for the given action
	 * 
	 * @param action
	 *            The action whose animations we should check
	 * @return A set of all supported directions
	 */
	public Set<? extends Direction> getDirections(String action) {
		if (this.animations.containsKey(action)) {
			return this.animations.get(action).keySet();
		}
		return Collections.emptySet();
	}

	/**
	 * Clones (i.e., copies) the current spritesheet, including contents
	 * 
	 * @return
	 */
	@Override
	public Spritesheet clone() {
		ColorModel cmodel = image.getColorModel();
		WritableRaster raster = image.copyData(null);
		Spritesheet retval = new Spritesheet(new BufferedImage(cmodel, raster, cmodel.isAlphaPremultiplied(), null),
				this.dyeChannels);
		retval.animations = this.animations;
		retval.setCelDimensions(celWidth, celHeight);
		return retval;
	}

	public Spritesheet clone(BufferedImage bf) {
		Spritesheet retval = new Spritesheet(bf, this.dyeChannels);
		retval.animations = this.animations;
		retval.setCelDimensions(celWidth, celHeight);
		return retval;
	}

	/**
	 * Retrieves a particular cel image
	 * 
	 * @param celNr
	 *            Number of the cel image
	 * @return The cel image, or null if celNr is invalid
	 */
	public Image getCel(int celNr) {
		if (celNr < 0 || celNr >= celColumns * celRows) {
			return null;
		}
		int xgrid = celNr % celColumns;
		int ygrid = celNr / celColumns;
		return image.getSubimage(celWidth * xgrid, celHeight * ygrid, celWidth, celHeight);
	}

	/**
	 * Starts an animation of the given action and direction
	 * 
	 * @param action
	 *            The action to animate
	 * @param direction
	 *            The direction to animate in
	 * @return A state object that allows us to determine the cel numbers and
	 *         the delays between cel times
	 */
	public AnimationState startAction(String action, Direction direction) {
		if (!this.animations.containsKey(action)) {
			return null;
		}
		Map<Direction, Animation> directions = this.animations.get(action);
		if (!directions.containsKey(direction)) {
			direction = Direction.DEFAULT;
		}
		if (!directions.containsKey(direction)) {
			return null;
		}
		return directions.get(direction).newState();
	}

	/**
	 * Prints useful information about the spritesheet in string form
	 */
	@Override
	public String toString() {
		return "Spritesheet(" + this.image.getWidth() + "x" + this.image.getHeight() + ", celSize=" + celWidth + "x"
				+ celHeight + ", celGrid = " + celColumns + "x" + celRows + ", dyeChannels=" + this.dyeChannels
				+ ", actions=" + getActions() + ")";
	}

	/**
	 * @return Image des Spritesheets
	 */
	public Image getImage() {
		return image;
	}

	/**
	 * @param img ist das Bild, das in das Spritesheet gespeichert werden soll
	 */
	public void setImage(Image img) {
		this.image = (BufferedImage) img;
	}

	/**
	 * @param c
	 *            ist die Farbe, zu der das Image des Spritesheets gefärbt
	 *            werden soll
	 * @param perc
	 *            ist die Intensität; 255 führt zum Informationsverlust der
	 *            Kontraste, da alle Pixel auf die gleiche Farbe gesetzt werden
	 * @param dye
	 *            ist der dyeChannel, der eingefärbt werden soll
	 */
	public void dye(Color c, int perc, String dye) {

			int r, g, b, a, argb = 0; 
			BufferedImage bf = new BufferedImage(this.image.getWidth(), this.image.getHeight(),
					BufferedImage.TYPE_INT_ARGB);
			String myDyes = this.getDyes().toString();

			if (myDyes.contains(dye) && dye == "WHITE") {		//soll der verfügbare weisse Dye-Kanal eingefärbt werden
				for (int y = 0; y < this.image.getHeight(); y++) {
					for (int x = 0; x < this.image.getWidth(); x++) {	//laufe durch alle Pixel des Images des Spritesheets
						argb = image.getRGB(x, y);	//speichere die aktuelle Farbe in argb
						r = (argb) & 0xFF;		//bit-shifts um die einzelnen Farben aus argb herauszuholen
						g = (argb >> 8) & 0xFF;
						b = (argb >> 16) & 0xFF;
						a = (argb >> 24) & 0xFF;	//hier kommt der Alpha-Wert ins Spiel, damit nicht transparente Pixel eingefärbt werden
						Color current = new Color(r, g, b, a); 		//erzeuge die aktuelle Farbe als Variable
						if (a != 0)		//ist der Pixel nicht-transparent?
							if ((r == b) && (b == g)) {		//r==b==g? -> gelesenes Pixel ist ein Weißton, färbe ihn ein
								int help = Gradient.getColor(current, c, perc).getRGB();	//erzeuge neue Farbe gemäß Gradienten und Intensität
								bf.setRGB(x, y, help);		//setze den aktuellen Pixel auf die neue Farbe
							} else {
								bf.setRGB(x, y, image.getRGB(x, y));	//lasse alle anderen Pixel unangerührt
							}
					}
				}
				this.setImage(bf);		//speichere das gefärbte Bild in das Spritesheet
				return;
			}

			if (myDyes.contains(dye) && dye == "BLUE") {	//analog zum Färben des Weissen Dye-Channels
				for (int y = 0; y < this.image.getHeight(); y++) {
					for (int x = 0; x < this.image.getWidth(); x++) {
						argb = image.getRGB(x, y);
						r = (argb) & 0xFF;
						g = (argb >> 8) & 0xFF;
						b = (argb >> 16) & 0xFF;
						a = (argb >> 24) & 0xFF;
						Color current = new Color(r, g, b, a);
						if (a != 0)
							if ((g == r) && (r == 0)) {	//Pixel ist Blau?
								int help = Gradient.getColor(current, c, perc).getRGB();
								bf.setRGB(x, y, help);
							} else {
								bf.setRGB(x, y, image.getRGB(x, y));
							}
					}
				}
				this.setImage(bf);
				return;
			}

			if (myDyes.contains(dye) && dye == "GREEN") {
				for (int y = 0; y < this.image.getHeight(); y++) {
					for (int x = 0; x < this.image.getWidth(); x++) {
						argb = image.getRGB(x, y);
						r = (argb) & 0xFF;
						g = (argb >> 8) & 0xFF;
						b = (argb >> 16) & 0xFF;
						a = (argb >> 24) & 0xFF;
						Color current = new Color(r, g, b, a);
						if (a != 0)
							if ((r == b) && (b == 0)) {	//Pixel grün?
								int help = Gradient.getColor(current, c, perc).getRGB();
								bf.setRGB(x, y, help);
							} else {
								bf.setRGB(x, y, image.getRGB(x, y));
							}
					}
				}
				this.setImage(bf);
				return;
			}

			if (myDyes.contains(dye) && dye == "RED") {
				for (int y = 0; y < this.image.getHeight(); y++) {
					for (int x = 0; x < this.image.getWidth(); x++) {
						argb = image.getRGB(x, y);
						r = (argb) & 0xFF;
						g = (argb >> 8) & 0xFF;
						b = (argb >> 16) & 0xFF;
						a = (argb >> 24) & 0xFF;
						Color current = new Color(r, g, b, a);
						if (a != 0)
							if ((r == g) && (g == 0)) {	//Pixel rot?
								int help = Gradient.getColor(current, c, perc).getRGB();
								bf.setRGB(x, y, help);
							} else {
								bf.setRGB(x, y, image.getRGB(x, y));
							}
					}
				}
				this.setImage(bf);
				return;
			}

			if (myDyes.contains(dye) && dye == "CYAN") {
				for (int y = 0; y < this.image.getHeight(); y++) {
					for (int x = 0; x < this.image.getWidth(); x++) {
						argb = image.getRGB(x, y);
						r = (argb) & 0xFF;
						g = (argb >> 8) & 0xFF;
						b = (argb >> 16) & 0xFF;
						a = (argb >> 24) & 0xFF;
						Color current = new Color(r, g, b, a);
						if (a != 0)
							if (r == 0) {	//Pixel cyan?
								int help = Gradient.getColor(current, c, perc).getRGB();
								bf.setRGB(x, y, help);
							} else {
								bf.setRGB(x, y, image.getRGB(x, y));
							}
					}
				}
				this.setImage(bf);
				return;
			}

			if (myDyes.contains(dye) && dye == "MAGENTA") {
				for (int y = 0; y < this.image.getHeight(); y++) {
					for (int x = 0; x < this.image.getWidth(); x++) {
						argb = image.getRGB(x, y);
						r = (argb) & 0xFF;
						g = (argb >> 8) & 0xFF;
						b = (argb >> 16) & 0xFF;
						a = (argb >> 24) & 0xFF;
						Color current = new Color(r, g, b, a);
						if (a != 0)
							if (g == 0) {	//Pixel magenta?
								int help = Gradient.getColor(current, c, perc).getRGB();
								bf.setRGB(x, y, help);
							} else {
								bf.setRGB(x, y, image.getRGB(x, y));
							}
					}
				}
				this.setImage(bf);
				return;
			}

			if (myDyes.contains(dye) && dye == "YELLOW") {
				for (int y = 0; y < this.image.getHeight(); y++) {
					for (int x = 0; x < this.image.getWidth(); x++) {
						argb = image.getRGB(x, y);
						r = (argb) & 0xFF;
						g = (argb >> 8) & 0xFF;
						b = (argb >> 16) & 0xFF;
						a = (argb >> 24) & 0xFF;
						Color current = new Color(r, g, b, a);
						if (a != 0)
							if (b == 0) {	//Pixel gelb?
								int help = Gradient.getColor(current, c, perc).getRGB();
								bf.setRGB(x, y, help);
							} else {
								bf.setRGB(x, y, image.getRGB(x, y));
							}
					}
				}
				this.setImage(bf);
				return;
			}
	}

}
