package de.uni_frankfurt.prgpr.core;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.Socket;
import java.util.concurrent.ConcurrentLinkedQueue;


/**
 * A socket-based channel interface that queues incoming messages and runs a thread for
 * reading them out.
 * 
 * @author creichen
 *
 */
public class QueuedChannel implements ChannelInterface {
	
	private Socket socket;
	private boolean connectionClosed = false;
	private ObjectOutputStream ostream;
	private ObjectInputStream istream;
	
	ConcurrentLinkedQueue<Serializable> receivedMessages = new ConcurrentLinkedQueue<>();

	/**
	 * Creates a new channel interface to communicate over the given socket
	 * 
	 * @param sock The socket to connect over
	 * @param istr object input stream, if already created
	 * @param ostr object output stream, if already created
	 */
	public QueuedChannel(Socket sock, ObjectInputStream istr, ObjectOutputStream ostr) {
		this.socket = sock;
		if (ostr != null) {
			ostream = ostr;
		} else try {
			ostream = new ObjectOutputStream(socket.getOutputStream());
		} catch (IOException exn) {
			disconnect();
			return;
		}
		istream = istr;
		
		new Thread(new ReceiverThread()).start();
	}
	
	@Override
	public void send(Serializable obj) {
		try {
			ostream.writeObject(obj);
			ostream.flush();
		} catch (IOException exn) {
			disconnect();
		}
	}
	
	@Override
	public Serializable receiveNext() {
		return receivedMessages.poll();
	}
	
	@Override
	public boolean isDisconnected() {
		return this.connectionClosed;
	}

	/**
	 * Keeps polling the input stream for new server-side messages
	 * @author creichen
	 *
	 */
	private class ReceiverThread implements Runnable {

		@Override
		public void run() {
			try {
				if (istream == null) {
					istream = new ObjectInputStream(socket.getInputStream());
				}
				while (true) {
					try {
						Serializable msg = (Serializable) istream.readObject();
						receivedMessages.add(msg);
					} catch (ClassNotFoundException e) {
						e.printStackTrace();
					}
				}
			} catch (IOException exn) {
				// connection broke down
				synchronized (QueuedChannel.this) {
					disconnect();
				}
			}
		}
	}

	@Override
	public void disconnect() {
		if (this.connectionClosed) {
			return;
		}
		try {
			socket.close();
		} catch (IOException e) {
		}
		connectionClosed = true;
	}
}
