package de.uni_frankfurt.prgpr.core;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.Socket;


/**
 * Client implementation; a class for sending messages
 * @author creichen
 *
 */
public class Client {
	/**
	 * Port to connect to
	 */
	private int port = Config.SERVER_PORT;

	/**
	 * Server to connect to
	 */
	private String server = null;
	/**
	 * Creates a new client that connects to the specified server
	 * @param serverName Server to connect to (IP address or DNS name)
	 */
	public Client(String serverName) {
		this.server = serverName;
	}
	
	/**
	 * Updates the server port number to connect to, from the default
	 */
	public void setPort(int port_nr) {
		this.port = port_nr;
	}
	
	/**
	 * Sends a synchronous message to the server and waits for a reply
	 * @param m The message to send
	 * @return The returned message
	 * @throws IOException If the network connection fails 
	 * @throws RuntimeException If the message didn't have all mandatory properties set 
	 */
	public Message sendMessage(Message m) throws IOException {
		m.validate();
		Message[] result = new Message[1];
		Socket sock = null;
		try {
			System.out.println("Trying to send a message.");
			sock = commWithServer(m, result);
		} finally {
			System.out.println("Success. Closing connection.");
			if (sock != null) {
				sock.close();
			}
		}
		return result[0];
	}

	ObjectOutputStream ostream = null;
	ObjectInputStream istream = null;
	
	/**
	 * Communicates a message to the server
	 * 
	 * @param sendMessage The message to send
	 * @param receiveMessage Array into which the received message will be written
	 * @return The communications socket
	 * @throws IOException If the network connection failed
	 */
	private Socket commWithServer(Message sendMessage, Message[] receiveMessage) throws IOException {
		Socket clientSocket = new Socket(server, port);
		ostream = new ObjectOutputStream(clientSocket.getOutputStream());
		ostream.writeObject(sendMessage);
		istream = new ObjectInputStream(clientSocket.getInputStream());
		try {
			Object obj = istream.readObject();
			if (obj instanceof Message) {
				receiveMessage[0] = (Message) obj;
			} else {
				receiveMessage[0] = new Message.Error("Invalid server reply: " + obj);
			}
		} catch (ClassNotFoundException exn) {
			// We're ONLY supposed to send `Message' objects and primitive types, so this should not ever happen
			// unless someone changed the Message class or the server implementation in an incompatible way.
			throw new RuntimeException(exn);
		}
		return clientSocket;
	}

	/**
	 * Establishes
	 * @param projectName
	 * @return
	 * @throws IOException
	 */
	public ChannelInterface connect(String projectName) throws IOException {
		Message[] result = new Message[1];
		Socket sock = null;
		
		Message.Connect mc = new Message.Connect();
		mc.setProject(projectName);
		sock = commWithServer(mc, result);

		if (result[0] instanceof Message.OK) {
			System.out.println("connected");
			return new QueuedChannel(sock, istream, ostream);
		} else {
			System.err.println("Failed to establish connection: " + result[0]);
		}
		return null;
	}
}
